/* Tetris - Windows Interface *\
\*     Lukas Turek, 2004      */

//#define UNICODE  //Program lze zkompilovat pro UNICODE,
//#define _UNICODE //ale pak nefunguje ve Windows 98

#include <windows.h>
#include <tchar.h>
#include <stdio.h>
#include <time.h>

#include "tetris.h"
#include "Tetr_win.h"

#define BLOCK_SIZE         16 //Velikost ctverecku (v pixelech)
#define STRING_MAX        100 //Delka retezcu nacitanych se zdroju
#define SPEED_RATIO      9/10 //O kolik se hra zrychluje
#define DEFAULT_LIMIT    1000 /* Vychozi hodnoty pro rychlost padani kostky     */
#define DEFAULT_ACCEL      10 /* a pocet dopadnutych kostek, nez se hra zrychli */
#define DEFAULT_INTENS      5 //Vychozi intenzita pro vbr kostky dle vyhodnosti

#define WM_USER_RESET    (WM_USER + 1) //Nova hra: wParam - typ hry, lParam - cislo hrace
#define WM_USER_GAMEOVER (WM_USER + 2) //Konec hry
#define WM_USER_BRICKSET (WM_USER + 3) //Kostka byla polozena: wParam - kostka (T_BRICK *)

//Vykresleni ctverecku se souradnicemi X,Y
#define DrawBlock(x,y) Rectangle(hdc, (x) * BLOCK_SIZE, (y) * BLOCK_SIZE, ((x) + 1) * BLOCK_SIZE - 1, ((y) + 1) * BLOCK_SIZE - 1);

//Souhrnne informace o hre (udaje o hre, vybrana kostka, pozice kostky a napovedy)
typedef struct { T_GAME game; T_BRICK *brick; T_COORD c, c_help; } T_GAMEDATA;

const T_COORD DEFCOORD = { 0, 4, 0 }; //Vychozi pozice kostky

//Texty: jmeno programu, chybove hlaseni, oznameni konce hry, text u rychlosti, soubor s manualem
TCHAR szProgramName[STRING_MAX], szReadError[STRING_MAX], szGameOver[STRING_MAX],
      szSpeed[STRING_MAX], szSettingsError[STRING_MAX], szManualFile[STRING_MAX];

T_BRICK bricks[7]; //Data o kostkach
unsigned int limit = DEFAULT_LIMIT; //Cas, nez kostka spadne (v milisekundach)
unsigned int orig_limit = DEFAULT_LIMIT; //Limit nastaveny uzivatelem (pred zrychlovanim)
unsigned int accel = DEFAULT_ACCEL; //Po kolika dopadnutych kostkach se zrychluje hra
unsigned int brsel_mode = BRSEL_RANDOM; //Zpusob, jak se vybira kostka
unsigned int brsel_intens = DEFAULT_INTENS; //Kolik je vyber kostky ovlivnen stavem hry
HWND hwndLeft, hwndRight, hwndStats; //Leve a prave herni okno, okno se stavovymi informacemi
int nStatsWidth; //Sirka stavoveho okna


//Obsluzna funkce pro dialog na volbu rychlosti
BOOL CALLBACK SettingsDlgProc(HWND hwnd, UINT nMsg, WPARAM wParam, LPARAM lParam)
{
  int new_limit, new_accel, new_intens; //Nove nastavene hodnoty

  switch(nMsg)
  {
    case WM_INITDIALOG: //Zapisuje se puvodni rychlost (v setinach sekundy) a zrychleni
      SetDlgItemInt(hwnd, IDD_SPEED, orig_limit / 10, FALSE);
      SetDlgItemInt(hwnd, IDD_ACCEL, accel, FALSE);

      //Zapise se intenzita pro vyber kostky a vybere se zvoleny druh vyberu kostky
      SetDlgItemInt(hwnd, IDD_INTENS, brsel_intens, FALSE);
      CheckRadioButton(hwnd, IDD_BRSEL1, IDD_BRSEL3, IDD_BRSEL1 + brsel_mode);
      return TRUE;

    case WM_COMMAND: //Zprava od tlacitka
      switch(LOWORD(wParam))
      {
        case IDOK: //Klepnuti na OK
          //Nacita se rychlost (v milisekundach) a zrychleni
          new_limit = GetDlgItemInt(hwnd, IDD_SPEED, NULL, FALSE) * 10;
          new_accel = GetDlgItemInt(hwnd, IDD_ACCEL, NULL, FALSE);
          new_intens = GetDlgItemInt(hwnd, IDD_INTENS, NULL, FALSE);

          //Zadna hodnota nesmi byt, limit muse byt maximalne minuta
          if(new_limit == 0 || new_accel == 0 || new_intens == 0 || new_limit > 60000)
            MessageBox(hwnd, szSettingsError, szProgramName, MB_ICONERROR);
          else
          {
            limit = limit * new_limit / orig_limit; //Limit klesne v pomeru, v jakem byl zmenen
            orig_limit = new_limit;    /* Nove hodnoty nastaveni se ulozi,  */
            accel = new_accel;         /* pro stanoveni druhu vyberu kostky */
            brsel_intens = new_intens; /* se musi projit vsechna tlacitka   */
            for(brsel_mode = 0; !IsDlgButtonChecked(hwnd, brsel_mode + IDD_BRSEL1); brsel_mode++);
            set_bricksel_mode(brsel_mode, brsel_intens); //Zpusob vyberu kostky se nastavi
            EndDialog(hwnd, IDOK); //Zavreni dialogu
          }
          break;

        case IDCANCEL: //Klepnuti na Storno
          EndDialog(hwnd, IDOK); //Zavreni dialogu
          break;
      }
      return TRUE; //Oznameni, e byla zprava zpracovana

    case WM_CLOSE: //Zavreni dalogu
      EndDialog(hwnd, 0);
      return TRUE;

    default: return FALSE; //Zprava nebyla zpracovana
  }
}


//Obsluzna funkce pro dialog "O programu"
BOOL CALLBACK AboutDlgProc(HWND hwnd, UINT nMsg, WPARAM wParam, LPARAM lParam)
{
  switch(nMsg)
  {
    case WM_INITDIALOG: /* Musi byt - jinak by se    */
      return TRUE;      /* neaktivovalo tlacitko OK! */

    case WM_CLOSE: //Zavreni dalogu
      EndDialog(hwnd, 0);
      return TRUE;

    case WM_COMMAND: //Tlacitko OK take zavre dialog
      if(LOWORD(wParam) == IDOK)
        EndDialog(hwnd, 0);
      return TRUE;

    default: return FALSE; //Zprava nebyla zpracovana
  }
}


//Vykresli ctverecky mezi (x1,y1) a (x2,y2) z pole BOARD do HDC
void DrawBoard(HDC hdc, T_BOARD board, int x1, int y1, int x2, int y2)
{
  int x,y; //Indexy cyklu - radek, sloupec
  int nBrush; //Zvoleny stetec (cislo standardniho objeku)

  if(y2 > 24) y2 = 24; /* Orezani na    */
  if(x2 > 11) x2 = 11; /* velikost pole */

  for(y = y1; y <= y2; y++)
    for(x = x1; x <= x2; x++)
    { //Prazdne ctverecky jsou bile (i okraj), ramecek tmave sedy, zbytek svetle sedy (a cerny okraj)
      nBrush = board[x][y] ? ((x == 0 || x == 11 || y == 24) ? GRAY_BRUSH : LTGRAY_BRUSH) : WHITE_BRUSH;
      SelectObject(hdc, GetStockObject(board[x][y] ? BLACK_PEN : WHITE_PEN));
      SelectObject(hdc, GetStockObject(nBrush));
      DrawBlock(x,y);
    }
}


//Vykresli kostku BRICK na souradnicich a natoceni C do HDC
void DrawBrick(HDC hdc, T_BRICK *brick, T_COORD *c)
{
  int i; //Index cyklu
  T_SHAPE *shape = &brick->rot[c->r]; //Vyber natoceni kostky

  SelectObject(hdc, GetStockObject(BLACK_PEN)); //Cerny obrys, vypln jiz byla zvolena
  for(i = 0; i < 4; i++) //Kresleni ctverecku kostky
    DrawBlock(shape->sq[i].x + c->x, shape->sq[i].y + c->y);
}


//Vykresli obsah herniho okna HWND pomoci udaju struktury G
void DrawWindow(HWND hwnd, T_GAMEDATA *g)
{
  HDC hdc; //Device Context
  PAINTSTRUCT ps; //Paint Structure
  RECT rect; //Ctverec, ktery se ma vykreslit

  if(GetUpdateRect(hwnd, &rect, TRUE)) //Zjistime, co mame kreslit
  {
    hdc = BeginPaint(hwnd, &ps);

    //Kresli se ctverecky, ktere jsou zcasti nebo zcela ve ctverci
    DrawBoard(hdc, g->game.board, rect.left / BLOCK_SIZE, rect.top / BLOCK_SIZE,
                                 (rect.right - 1) / BLOCK_SIZE, (rect.bottom - 1) / BLOCK_SIZE);

    if(g->game.mode == GAME_HELP) /* Vykresleni napovedy        */
    {                             /* (ctverecky s bilou vyplni) */
      SelectObject(hdc, GetStockObject(WHITE_BRUSH));
      DrawBrick(hdc, g->brick, &g->c_help);
    }

    if(g->brick) /* Vykresleni kostky, pokud existuje */
    {            /* (ctverecky s cernou vyplni)       */
      SelectObject(hdc, GetStockObject(BLACK_BRUSH));
      DrawBrick(hdc, g->brick, &g->c);
    }
    EndPaint(hwnd, &ps);
  }
}


//Oznaci ze se ma preklreslit kostka BRICK na souradnicich C v okne HWND
void InvalidateBrick(HWND hwnd, T_BRICK *brick, T_COORD *c)
{
  RECT rect; //Ctverec ohranicujici kostku
  T_SHAPE *shape = &brick->rot[c->r]; //Vyber natoceni kostky

  //Pouziji se hranice, ktere jsou pro kostku predpocitany
  rect.left = (shape->x1 + c->x) * BLOCK_SIZE;
  rect.top = (shape->y1 + c->y) * BLOCK_SIZE;
  rect.right = (shape->x2 + c->x + 1) * BLOCK_SIZE;
  rect.bottom = (shape->y2 + c->y + 1) * BLOCK_SIZE;
  InvalidateRect(hwnd, &rect, FALSE); //Ctverec ohranicujici kostku se prekresli
}


//Oznaci, ze se maji prekreslit rady mezi Y1 a Y2 (vcetne) v okne HWND
void InvalidateRows(HWND hwnd, int y1, int y2)
{
   RECT rect; //Ctverec ohranicujici prekreslovane rady

   rect.left = BLOCK_SIZE;       /* Rady se prekresluji bez */
   rect.right = 11 * BLOCK_SIZE; /* ramecku, ten se nemeni  */
   rect.top = y1 * BLOCK_SIZE;
   rect.bottom = (y2 + 1) * BLOCK_SIZE;
   InvalidateRect(hwnd, &rect, FALSE);
}


//Aktualizuje se stavove informace v okne hwndStats dle udaju struktury GAME
void ShowStats(HWND hwndStats, T_GAME *game)
{ //Napred se vypisuji data o vymazanych radach (u kazdeho hrace jine)
  if(game->mode == GAME_MATCH && game->player == 0)
  { //Doleva se kresli pouze udaje hrace 0 pokud hraji dva hraci
    SetDlgItemInt(hwndStats, IDD_ALLROWS_0, game->stats.all_rows, FALSE);
    SetDlgItemInt(hwndStats, IDD_ROWS1_0, game->stats.rows_by_number[1], FALSE);
    SetDlgItemInt(hwndStats, IDD_ROWS2_0, game->stats.rows_by_number[2], FALSE);
    SetDlgItemInt(hwndStats, IDD_ROWS3_0, game->stats.rows_by_number[3], FALSE);
    SetDlgItemInt(hwndStats, IDD_ROWS4_0, game->stats.rows_by_number[4], FALSE);
  }
  else /* Udaje hrace 1 se vykresli doprava, totez  */
  {    /* pokud hraje jen jeden hrac, vypada to lip */
    SetDlgItemInt(hwndStats, IDD_ALLROWS_1, game->stats.all_rows, FALSE);
    SetDlgItemInt(hwndStats, IDD_ROWS1_1, game->stats.rows_by_number[1], FALSE);
    SetDlgItemInt(hwndStats, IDD_ROWS2_1, game->stats.rows_by_number[2], FALSE);
    SetDlgItemInt(hwndStats, IDD_ROWS3_1, game->stats.rows_by_number[3], FALSE);
    SetDlgItemInt(hwndStats, IDD_ROWS4_1, game->stats.rows_by_number[4], FALSE);
  }

  if(game->mode != GAME_MATCH) /* Pokud hraje jen jeden hrac, */
  {                            /* jsou udaje vlevo prazdne    */
    SetDlgItemText(hwndStats, IDD_ALLROWS_0, TEXT (""));
    SetDlgItemText(hwndStats, IDD_ROWS1_0, TEXT (""));
    SetDlgItemText(hwndStats, IDD_ROWS2_0, TEXT (""));
    SetDlgItemText(hwndStats, IDD_ROWS3_0, TEXT (""));
    SetDlgItemText(hwndStats, IDD_ROWS4_0, TEXT (""));
  }

  if(game->player == 0) /* Struktura prvniho hrace ma v sobe i udaje o poctu kostek  */
  {                     /* (u druheho hrace jsou totozne a neukladaji se duplicitn) */
    SetDlgItemInt(hwndStats, IDD_ALLBRICKS, game->stats.all_bricks, FALSE);
    SetDlgItemInt(hwndStats, IDD_BRICKS0, game->stats.single_brick[0], FALSE);
    SetDlgItemInt(hwndStats, IDD_BRICKS1, game->stats.single_brick[1], FALSE);
    SetDlgItemInt(hwndStats, IDD_BRICKS2, game->stats.single_brick[2], FALSE);
    SetDlgItemInt(hwndStats, IDD_BRICKS3, game->stats.single_brick[3], FALSE);
    SetDlgItemInt(hwndStats, IDD_BRICKS4, game->stats.single_brick[4], FALSE);
    SetDlgItemInt(hwndStats, IDD_BRICKS5, game->stats.single_brick[5], FALSE);
    SetDlgItemInt(hwndStats, IDD_BRICKS6, game->stats.single_brick[6], FALSE);
  }
}


//Vypis rychlosti hry do stavoveho okna (jako retezec, aby se vycentrovalo)
void ShowSpeed(int limit)
{
  TCHAR buffer[256];

  _sntprintf(buffer, 256, TEXT ("%s %d"), szSpeed, limit / 10);
  SetDlgItemText(hwndStats, IDD_GAMESPEED, buffer);
}


//Nastaveni velikosti okna HWND: CX bodu vertikalne, CY bodu horizontalne
void SetWindowSize(HWND hwnd, int cx, int cy)
{
  RECT rcWin, rcClient; //Rozmer okna, rozmer klientske oblasti
  int cxSize, cySize; //Vypocitana velikost okna

  GetClientRect(hwnd, &rcClient); /* Odectenim velikosti okna a klientske oblasti     */
  GetWindowRect(hwnd, &rcWin);    /* se zjisti, o kolik musi byt okno vetsi nez CX,CY */
  cxSize = rcWin.right - rcWin.left - rcClient.right + cx;
  cySize = rcWin.bottom - rcWin.top - rcClient.bottom + cy;
  MoveWindow(hwnd, rcWin.left, rcWin.top, cxSize, cySize, TRUE);
}


//Vybere novou kostku z pole BRICKS pro hru G v okne HWND
void GetNewBrick(HWND hwnd, T_BRICK *bricks, T_GAMEDATA *g)
{
  if(g->game.player == 0) //Kostku vybira pouze prvni hrac
    g->brick = new_brick(bricks, &g->game); /* Nahodny vyber kostky,       */
                                            /* aktualizace stavovych udaju */
  if(g->game.mode == GAME_COMP || g->game.player == 1) /* Hraje-li pocitac,    */
    find_pos(g->brick, &g->c, &g->game);               /* vybere pozici pro    */
  else                                                 /* kostku, jinak se     */
    g->c = DEFCOORD;                                   /* zvoli vychozi pozice */

  if(g->game.mode == GAME_HELP) /* Ma-li se zobrazovat napoveda, vybere se */
  {                             /* pozice pro kostku a napoveda se zobrazi */
    find_pos(g->brick, &g->c_help, &g->game);
    InvalidateBrick(hwnd, g->brick, &g->c_help);
  }

  if(!(g->game.stats.all_bricks % accel)) //Po dopadu ACCEL kostek se hra zrychluje
  {
    limit = limit * SPEED_RATIO; //Doba mezi pady kostky klesne na SPEED_RATIO puvodni doby
    ShowSpeed(limit); //Aktualni rychlost se zobrazi
    KillTimer(GetParent(hwnd), 1); //A pak se nastavi
    SetTimer(GetParent(hwnd), 1, limit, NULL);
  }
  
  InvalidateBrick(hwnd, g->brick, &g->c); //Kostka se zobrazi
  ShowStats(hwndStats, &g->game); //Zobrazi se stavove udaje
}


//Zpracovani vstupu od cloveka v okne HWND: podle klavesy KEY se upravi
//pozice C pro kostku BRICK a aktualizuji se herni udaje ve strukture GAME
int HumanInput(HWND hwnd, int key, T_GAME *game, T_BRICK *brick, T_COORD *c)
{
  int command; //Klavesa prevedena na prikaz

  switch(key) //Prevod klavesy na prikaz pro funkci move_window
  {
    case VK_LEFT:   command = KEY_LEFT;  break; //Doleva
    case VK_RIGHT:  command = KEY_RIGHT; break; //Doprava
    case VK_DOWN:   command = KEY_DOWN;  break; //Dolu
    case 'Z':       command = KEY_RROT;  break; //Otoceni po smeru hodinovych rucicek
    case 'X':       command = KEY_LROT;  break; //Otoceni proti smeru hodinovych rucicek
    case VK_RETURN: command = KEY_FALL;  break; //Pad kostky
    default: return 1;
  }

  InvalidateBrick(hwnd, brick, c); //Smaze se predchozi pozice kostky
  if(!move_brick(command, c, brick, game->board)) /* Pokud kostka po provedeni */
  {                                               /* prikazu dopadla           */
    if(!set_brick(&brick->rot[c->r], c, game)) /* Kostka se polozi, zustala-li */
    {                                          /* ve vychozi pozici, hra konci */
      SendMessage(GetParent(hwnd), WM_USER_GAMEOVER, 0, 0);
      return 1;
    }
    else
    {
      InvalidateBrick(hwnd, brick, c); //Vykresluje se nova pozice kostky
      if(game->last_changed != -1) //Pokud se rady zmenily, vykresli se
        InvalidateRows(hwnd, game->first_changed, game->last_changed);
      return 0;
    }
  }

  InvalidateBrick(hwnd, brick, c); //Vykresluje se nova pozice kostky (nebo nova kostka)
  return 1;
}

//Pocitac vybere pozici C pro kostku BRICK v okne HWND, aktualizuji se herni udaje v GAME
int CompInput(HWND hwnd, T_GAME *game, T_BRICK *brick, T_COORD *c)
{
  InvalidateBrick(hwnd, brick, c); //Rusi se stara pozice kostky
  if(!set_brick(&brick->rot[c->r], c, game)) /* Kostka se polozi, zustala-li */
  {                                          /* ve vychozi pozici, hra konci */
    SendMessage(GetParent(hwnd), WM_USER_GAMEOVER, 0, 0);
    return 1;
  }
  else
    if(game->last_changed != -1) //Pokud se rady zmenily, vykresli se
      InvalidateRows(hwnd, game->first_changed, game->last_changed);
  return 0;
}


//Obsluzna funkce herniho okna (pro obe okna, v kazdem dostava jinou strukturu GAMEDATA)
LRESULT CALLBACK GameWndProc(HWND hwnd, UINT nMsg, WPARAM wParam, LPARAM lParam, T_GAMEDATA *gamedata)
{
  switch (nMsg)
  {
    case WM_CREATE:
      initialize(&gamedata->game); //Vytvori se ramecek okolo pole, aby bylo co kreslit
      gamedata->brick = NULL; //Nebyla vybrana zadna kostka
      return 0;

    case WM_USER_RESET: //Nova hra
      initialize(&gamedata->game); //Inicializace struktury GAME
      gamedata->game.mode = wParam; //Predany typ hry se zapise do struktury GAME
      gamedata->game.player = lParam; //Uklada se i cislo hrace

      if(lParam == 1) /* Kostku vybira jen prvni hrac, druhy si oznaci, ze zadna nebyla */
      {               /* vybrana a aktualizuje statistiku (nomalne to dela GetNewBrick) */
        ShowStats(hwndStats, &gamedata->game);
        gamedata->brick = NULL;
      }
      else
        GetNewBrick(hwnd, bricks, gamedata);

      InvalidateRows(hwnd, 0, 23); //Cele pole se prekresli (smaze stara hra)
      return 0;

    case WM_KEYDOWN: //Stisk klacesy preposlaby z MainWndProc
      switch(gamedata->game.mode) //V kazdem typu hry se musime chovat jinak
      {                  /* Standardni hra: provede se prikaz,   */
        case GAME_HUMAN: /* pokud kostka dopadla, vybere se nova */
          if(!HumanInput(hwnd, wParam, &gamedata->game, gamedata->brick, &gamedata->c))
            GetNewBrick(hwnd, bricks, gamedata);
          return 0;

        case GAME_COMP: //Hra pocitace: polozi se kostka a vybere nova
          if(!CompInput(hwnd, &gamedata->game, gamedata->brick, &gamedata->c))
            GetNewBrick(hwnd, bricks, gamedata);
          return 0;
                        /* Hra s napovedou: jako standardni hra, jen se musi mazat */
        case GAME_HELP: /* stara napoveda (zobrazovani nove udela GetNewBrick)     */
          if(!HumanInput(hwnd, wParam, &gamedata->game, gamedata->brick, &gamedata->c))
          {
            InvalidateBrick(hwnd, gamedata->brick, &gamedata->c_help);
            GetNewBrick(hwnd, bricks, gamedata);
          }
          return 0;
                         /* Hra dvou hracu: zpravu WM_KEYDOWN dostava jen  */
        case GAME_MATCH: /* prvni okno, a to funguje jako u standardni hry */
          if(!HumanInput(hwnd, wParam, &gamedata->game, gamedata->brick, &gamedata->c))
          {
            SendMessage(hwndRight, WM_USER_BRICKSET, (WPARAM) gamedata->brick, 0);
            GetNewBrick(hwnd, bricks, gamedata);
          }
          return 0;

        default: return 0;
      }

    case WM_USER_BRICKSET: //V levem okne byla polozena kostka, ma se polozit i v pravem
      if(gamedata->brick) //Pokladame predchozi kostku (pokud existuje)
        CompInput(hwnd, &gamedata->game, gamedata->brick, &gamedata->c);
      gamedata->brick = (T_BRICK *) wParam; //Kostka byla predana jako wParam
      GetNewBrick(hwnd, bricks, gamedata); //Nova kostka se zobrazi a vybere se pozice
      return 0;

    case WM_PAINT: //Vykresleni okna - pouze zde opravdu kreslime
      DrawWindow(hwnd, gamedata);
      return 0;

    default: return DefWindowProc(hwnd, nMsg, wParam, lParam);
  }
}


//Fukce pro prave a leve okno se lisi jen strukturou GAMEDATA
LRESULT CALLBACK LeftWndProc(HWND hwnd, UINT nMsg, WPARAM wParam, LPARAM lParam)
{
  static T_GAMEDATA gamedata;
  return GameWndProc(hwnd, nMsg, wParam, lParam, &gamedata);
}


//Fukce pro prave a leve okno se lisi jen strukturou GAMEDATA
LRESULT CALLBACK RightWndProc(HWND hwnd, UINT nMsg, WPARAM wParam, LPARAM lParam)
{
  static T_GAMEDATA gamedata;
  return GameWndProc(hwnd, nMsg, wParam, lParam, &gamedata);
}


//Vytvoreni hernich oken a okna pro stavove udaje v okne HWND
void CreateChildWindows(HWND hwnd)
{
  WNDCLASSEX wndclass; //Trida pro prave a leve okno
  RECT rect;

  memset(&wndclass, 0, sizeof(WNDCLASSEX));
  wndclass.cbSize = sizeof(WNDCLASSEX);
  wndclass.style         = CS_HREDRAW | CS_VREDRAW;
  wndclass.hInstance     = GetModuleHandle(NULL);
  wndclass.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);

  //Vytvoreni leveho okna (zcela vlevo, velke akorat na pole 12x25)
  wndclass.lpfnWndProc   = LeftWndProc;
  wndclass.lpszClassName = TEXT ("LeftWin");
  RegisterClassEx(&wndclass);
  hwndLeft = CreateWindow(TEXT ("LeftWin"), NULL, WS_CHILD | WS_VISIBLE,
                          0, 0, 12 * BLOCK_SIZE, 25 * BLOCK_SIZE,
                          hwnd, NULL, GetModuleHandle(NULL), NULL);

  //Stavove okno je jako dialog, aby se dalo natahnout ze zdroju
  hwndStats = CreateDialog(GetModuleHandle(NULL), MAKEINTRESOURCE(ID_STATS), hwnd, NULL);
  GetWindowRect(hwndStats, &rect);      /* Zjisteni sirky */
  nStatsWidth = rect.right - rect.left; /* stavoveho okna */
  //Stavove okno se posune vedle leveho okna a roztahne na celou vysku
  MoveWindow(hwndStats, 12 * BLOCK_SIZE, 0, nStatsWidth, 25 * BLOCK_SIZE, TRUE);
  ShowSpeed(orig_limit); //Vypisuje se rychlost do stavoveho okna

  //Vytvoreni praveho okna (vpravo vedle stavoveho okna, stejne velke)
  wndclass.lpfnWndProc   = RightWndProc;      /* Okna se lisi jen obsluznou funkci,  */
  wndclass.lpszClassName = TEXT ("RightWin"); /* ale kvuli tomu musi mit ruzne tridy */
  RegisterClassEx(&wndclass);
  hwndRight = CreateWindow(TEXT ("RightWin"), NULL, WS_CHILD | WS_VISIBLE,
                          12 * BLOCK_SIZE + nStatsWidth, 0, 12 * BLOCK_SIZE, 25 * BLOCK_SIZE,
                          hwnd, NULL, GetModuleHandle(NULL), NULL);

  //Nastaveni velikosti okna (zatim bude videt jen leve a stavove okno)
  SetWindowSize(hwnd, 12 * BLOCK_SIZE + nStatsWidth, 25 * BLOCK_SIZE);
}


//Spusti novou hru typu GAMETYPE v okne HWND
void StartGame(HWND hwnd, int gametype)
{
  limit = orig_limit; //Rusi se zrychleni od predchozi hry

  SendMessage(hwndLeft, WM_USER_RESET, gametype, 0); //Inicializuje se leve okno
  ShowSpeed(limit); //Zobrazi se rychlost hry

  if(gametype == GAME_MATCH) /* Pokud se hraje proti pocitaci, inicializuje se i prave */
  {                          /* okno a hlavni okno se zvetsi, aby bylo to prave videt  */
    SendMessage(hwndRight, WM_USER_RESET, GAME_MATCH, 1);
    SetWindowSize(hwnd, 24 * BLOCK_SIZE + nStatsWidth, 25 * BLOCK_SIZE);
  }
  else //Veliost hlavniho okna se nastavi, aby bylo videt leve okno a statistika
    SetWindowSize(hwnd, 12 * BLOCK_SIZE + nStatsWidth, 25 * BLOCK_SIZE);

  SetTimer(hwnd, 1, limit, NULL); //Kostka zacne padat
}


//Obsluzna funkce hlavniho okna programu
LRESULT CALLBACK MainWndProc(HWND hwnd, UINT nMsg, WPARAM wParam, LPARAM lParam)
{
  static BOOL bPaused = 0; //Zda je hra pozastavena (kostka nepada)
  static BOOL bStopped = 1; //Zda je hra zastavena (pred spustenim a po GameOver)

  switch (nMsg)
  {
    case WM_CREATE: //Pri vytvoreni se vytvori herni okna a stavove okno
      CreateChildWindows(hwnd);
      return 0;

    case WM_KEYDOWN: //Stisknuta klavesa
      if(bStopped) /* Je-li hra zastavena, */
        return 0;  /* klavesy se ignoruji  */

      if(bPaused && wParam != VK_SPACE) /* Zapauzovana hra se odpauzuje libovolnou */
      {                                 /* klavesou krome mezery (to je klavesova  */
        bPaused = 0;                    /* zkratka na pauzu, takze by se hra       */
        SetTimer(hwnd, 1, limit, NULL); /* zaroven zapauzovala i odpauzovala):     */
        return 0;                       /* nastavi se casovac, kostka zacne padat) */
      }

      /* Stisknuta klavesa se preposila levemu oknu (klavesy ESC a SPACE */
      /* jsou jako klavesove zkratky, takze se zpracuji pri WM_COMMAND)  */
      return SendMessage(hwndLeft, nMsg, wParam, lParam);

    case WM_TIMER:                                   /* Casovac (kostka ma spadnout): totez, */
      SendMessage(hwndLeft, WM_KEYDOWN, VK_DOWN, 0); /* jako by byla stisknota sipka dolu    */
      return 0;

    case WM_COMMAND: //Prikaz v menu
      switch(LOWORD(wParam))
      {
        case IDM_EXIT: //Konec hry - zavreni okna
          DestroyWindow(hwnd);
          return 0;

        case IDM_PAUSE: //Pozastaveni hry
          if(bStopped) /* Zastavena hra    */
            return 0;  /* nejde pozastavit */

          if(bPaused) //Je-li hra zapauzovana, odpauzuje se a naopak
          {
            bPaused = 0;
            SetTimer(hwnd, 1, limit, NULL);
          }
          else
          {
            bPaused = 1;
            KillTimer(hwnd, 1);
          }
          return 0;

        case IDM_SETTINGS:       /* Zmena rychlosti: zrusi se casovac (bude se menit),     */
          KillTimer(hwnd, 1);    /* zeptame se na rychlost a zobrazime ji ve stavovem okne */
          DialogBox(GetModuleHandle(NULL), MAKEINTRESOURCE(ID_SETTINGS), hwnd, SettingsDlgProc);

          if(bStopped) //Pokud hra nebezi, zobrazi se nastavena rychlsot
            ShowSpeed(orig_limit);
          else
          {
            ShowSpeed(limit); //Jinak se zobrazi prepocitana aktualni rychlost
            SetTimer(hwnd, 1, limit, NULL); //Znovu se zapne casovac (s novou rychlosti)
          }
          return 0;

        /* Zobrazeni manualu (HTML stranka se otevre ve vychozim       */
        /* prohlizeci). Hra se zapauzuje, jako by byla stisknuta pauza */
        case IDM_MANUAL:
          SendMessage(hwnd, WM_COMMAND, IDM_PAUSE, 0);
          ShellExecute(hwnd, TEXT ("open"), szManualFile, NULL, NULL, SW_SHOWNORMAL);
          return 0;

        case IDM_ABOUT:       /* Dialog "O programu": zobrazi se dialogov okno, */
          KillTimer(hwnd, 1); /* ale nesmime zapomenout vypnout casovac!         */
          DialogBox(GetModuleHandle(NULL), MAKEINTRESOURCE(ID_ABOUT), hwnd, AboutDlgProc);
          if(!bStopped) //Casovac se zapina jen bezi-li hra
            SetTimer(hwnd, 1, limit, NULL);
          return 0;

        /* Nova hra: zrusi se zastaveni hry, informuje se leve (u hry proti pocitaci */
        /* i prave) okno, nastavi se velikost okna (aby bylo videt leve okno,        */
        /* stavove okno a u hry proti pocitaci i prave okno) a zapne se casovac      */
        case IDM_NEWGAME_HUMAN: //Standardni hra
           bStopped = 0;
           StartGame(hwnd, GAME_HUMAN);
           return 0;

        case IDM_NEWGAME_COMP: //Automaticka hra
           bStopped = 0;
           StartGame(hwnd, GAME_COMP);
           return 0;

        case IDM_NEWGAME_HELP: //Hra s napovedou
           bStopped = 0;
           StartGame(hwnd, GAME_HELP);
           return 0;

        case IDM_NEWGAME_MATCH: //Hra proti pocitaci
           bStopped = 0;
           StartGame(hwnd, GAME_MATCH);
           return 0;

        default: return 0;
      }

    case WM_USER_GAMEOVER: /* Konec hry: zobrazi se zprava     */
      KillTimer(hwnd, 1);  /* a hra se zastavi (vypne casovac) */
      MessageBox(hwnd, szGameOver, szProgramName, MB_OK);
      bStopped = 1;
      return 0;

    case WM_DESTROY:
      PostQuitMessage(0);
      return 0;
  }
  return DefWindowProc(hwnd, nMsg, wParam, lParam);
}


//-=<MAIN>=-
int APIENTRY WinMain (HINSTANCE hInst, HINSTANCE hPrev, LPSTR lpCmd, int nShow)
{
  HWND hwndMain; //Hlavni okno
  HACCEL hAccel; //Klavesove zkratky
  MSG msg; //Prijata zprava
  WNDCLASSEX wndclass; //Trida hlavniho okna

  //Nacitani retezcu ze zdroju
  LoadString(hInst, IDS_PROGRAMNAME, szProgramName, STRING_MAX);
  LoadString(hInst, IDS_READERROR, szReadError, STRING_MAX);
  LoadString(hInst, IDS_GAMEOVER, szGameOver, STRING_MAX);
  LoadString(hInst, IDS_SPEED, szSpeed, STRING_MAX);
  LoadString(hInst, IDS_SETTINGSERROR, szSettingsError, STRING_MAX);
  LoadString(hInst, IDS_MANUALFILE, szManualFile, STRING_MAX);

  srand(time(NULL)); //Pro opravdu nahodne kostky
  if(!read_bricks(bricks)) //Nacteni a vypocet udaju o kostkach
    return MessageBox(NULL, szReadError, szProgramName, MB_ICONERROR);

  memset(&wndclass, 0, sizeof(WNDCLASSEX));
  wndclass.lpszClassName = szProgramName;
  wndclass.cbSize = sizeof(WNDCLASSEX);
  wndclass.style = CS_HREDRAW | CS_VREDRAW;
  wndclass.lpfnWndProc = MainWndProc;
  wndclass.hInstance = hInst;
  wndclass.hIcon = LoadIcon(hInst, MAKEINTRESOURCE(ID_ICON));
  wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
  wndclass.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
  wndclass.lpszMenuName = MAKEINTRESOURCE(ID_MENU);

  RegisterClassEx(&wndclass);
  hwndMain = CreateWindow(szProgramName, szProgramName,
                          WS_SYSMENU | WS_CAPTION | WS_MINIMIZEBOX,
                          CW_USEDEFAULT, CW_USEDEFAULT,
                          CW_USEDEFAULT, CW_USEDEFAULT,
                          NULL, NULL, hInst, NULL);

  ShowWindow(hwndMain, nShow);
  UpdateWindow(hwndMain);
  hAccel = LoadAccelerators (hInst, MAKEINTRESOURCE(ID_ACCEL)) ;

  while(GetMessage(&msg, NULL, 0, 0))
    if(!TranslateAccelerator(hwndMain, hAccel, &msg))
    {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }
  return msg.wParam;
}
