/**
 * \file mp3stream.h
 * @short Declaration of class MP3Stream
 */

#ifndef _MP3STREAM_H_
#define _MP3STREAM_H_

#include "mp3frame.h"
#include "bitstream.h"

/**
 * @short Helper class for MP3Frame
 *
 * Purpose of this class is to store information shared 
 * between MP3Frame objects. It stores pointers to first
 * frame whose data weren't written yet (<em>firstframe</em>)
 * and also to frame with those data (<em>dataframe</em>).
 */
class MP3Stream
{
  public:
    
    /**
     * Constructor
     *
     * Creates a MP3Stream on top of Bitstream,
     * reads a first frame.
     *
     * @param bs Input Bitstream to read data from
     * @param type Write mode as defined in MP3Frame
     */
    MP3Stream(Bitstream &bs, int type);
    
    /**
     * Destructor
     *
     * Deletes all allocated frames.
     */
    ~MP3Stream();
    
    /**
     * Returns \e firstframe of the stream. 
     *
     * @return First frame
     */
    MP3Frame *First() const;
    
    /**
     * Sets \e firstframe of the stream
     *
     * It does not delete any frames, because they can
     * store data of this new first frame.
     *
     * @param frame New first frame
     */
    void SetFirst(MP3Frame *frame);
    
    /**
     * Returns the associated Bitstream
     *
     * @return Input Bitstream
     */
    Bitstream &GetBitstream() const;
    
    /**
     * Writes requested number of data bytes from stream
     *
     * This is the most important function. It choses a frame whose 
     * data to write (more frames may be needed to fullfill the 
     * request) and calls its 
     * \link MP3Frame::WriteData() WriteData() \endlink function.
     *
     * @param bs Output bitstream 
     * @param numbytes Number of bytes to write
     * @param startpos Position of first byte in stream
     */
    void WriteData(Bitstream &bs, int numbytes, int startpos);
    
    /**
     * Sets a new <em>dataframe</em> deleting all frames before it
     *
     * It's called by MP3Frame::WriteData() function to delete all 
     * "emptied" \e dataframes (frames from which all data were written)
     *
     * @param frame New \e dataframe
     */
    void SeekTo(MP3Frame *frame);
    
    /**
     * Returns the write mode of stream
     *
     * This value is used by MP3Frame object and can be
     * MP3Frame::normal, MP3Frame::fastwrite or MP3Frame::nowrite.
     *
     * @return Write mode of stream
     */
    int WriteMode() const;
  
  private:
    Bitstream &m_stream; //Input bitsteam
    int m_cutbytes; //How many bytes were written from frame
    int m_type; //Write mode
    MP3Frame *m_f, *m_ff; //Firstframe, Dataframe
};

#endif //_MP3STREAM_H_
