/**
 * \file mp3frame.cpp
 * @short Implementation of class MP3Frame
 */

//#define DEBUG

#include <cassert>
#include "mp3frame.h"
#include "mp3stream.h"
#include "except.h"

#ifdef DEBUG
#include <iostream>
#endif

MP3Frame::MP3Frame(MP3Stream *stream, MP3Frame *prev)
{
  m_stream = stream;
  m_corruption = false;
  m_firstframe = (prev == 0); //First frame has no previous
  SetZeroes(); //Initialize everything else to zero
  
  while(!Init(prev)) //Until we don't find valid frame
  {                  // (or EOF via exception)
    Cleanup();
    SetZeroes();
  }
    
  #ifdef DEBUG
  std::cout << "newframe @" << m_totalbytes << ", size " << m_spacesize << ", data " << m_datasize << " begin -" << m_databegin << std::endl;
  #endif
}  

MP3Frame::~MP3Frame()
{
  Cleanup();
}  

void MP3Frame::Cleanup()
{
  delete m_fh;
  delete m_dh;
  delete m_ch[0][0];
  delete m_ch[0][1];
  delete m_ch[1][0];
  delete m_ch[1][1];
  delete m_data;
}  

void MP3Frame::SetZeroes()
{
  //Zeroes everywhere
  m_fh = 0;
  m_dh = 0;
  m_ch[0][0] = m_ch[0][1] = m_ch[1][0] = m_ch[1][1] = 0;
  m_data = 0;
  m_next = 0;
  m_spacesize = m_datasize = m_databegin = 0;
  m_framesize = m_newdatabegin = m_stuffing = 0;
  m_sizechange = 0;
  m_locked = false;
}

bool MP3Frame::Init(MP3Frame *prev)
{
  int headersize = mp3::syncbits; //Synchronization bits are not in data structure
  Bitstream &bs = m_stream->GetBitstream(); //Read operations work with bitstream
  
  try {
    int garbage = bs.Find(mp3::sync, mp3::syncbits); //Synchronize
    
    if(garbage) //If synchronization was needed
    {  
      #ifdef DEBUG
      std::cout << "corruption: garbage (" << garbage << ')' << std::endl;
      #endif
      
      m_corruption = !m_firstframe; //HACK: to skip ID3 tag at the beginning
    }
    
    m_fh = new FrameHeader(bs);        
    
    #ifdef DEBUG
    std::cout << "frameheader: V" << m_fh->Version() << " L" << m_fh->Layer() << ' ' << m_fh->Bitrate() << "kbps" << std::endl;
    #endif

    if(!m_fh->isValid()) //Check validity (format, layer, etc.)
    {  
      #ifdef DEBUG
      std::cout << "corruption: invalid frame header" << std::endl;
      #endif
      
      m_corruption = true;
      return false;
    }
    
    //TODO: Really check CRC (now i just store it)
    if(m_fh->isCRC()) 
    {  
      #ifdef DEBUG
      std::cout << "CRC code found" << std::endl;
      #endif
      
      bs.Read(&m_CRC, mp3::CRCbits);
      headersize += mp3::CRCbits; //Don't forget to count bits!
    }
    
    //Data headers differ by field sizes and positions
    if(m_fh->Version() == 1)
      if(m_fh->NumChannels() == 1)
        m_dh = new DataHeaderMPEG1Mono(bs);
      else
        m_dh = new DataHeaderMPEG1Stereo(bs);
    else
      if(m_fh->NumChannels() == 1)
        m_dh = new DataHeaderMPEG2Mono(bs);
      else
        m_dh = new DataHeaderMPEG2Stereo(bs);
    
    headersize += m_fh->Size() + m_dh->Size();
    m_framesize = m_fh->FrameSize();
    m_databegin = m_dh->DataBegin();
         
    //Read Channel Headers, count data
    for(int i = 0; i < m_fh->NumGranules(); i++)
      for(int j = 0; j < m_fh->NumChannels(); j++)
      {  
        if(m_fh->Version() == 1)
          m_ch[i][j] = new ChannelHeaderMPEG1(bs);
        else
          m_ch[i][j] = new ChannelHeaderMPEG2(bs);
        
        headersize += m_ch[i][j]->Size();
        m_datasize += m_ch[i][j]->DataSize();
      }
    
    m_datasize = (m_datasize + 7) >> 3; //The sizes are in bits
    m_spacesize = m_fh->FrameSize() - (headersize >> 3); //Header size too
    //Calculate the position in stream (after previous frame, if there's one)
    m_totalbytes = prev ? (prev->m_totalbytes + prev->m_spacesize) : 0;
    
    //Check if the frames don't overlap and if the frame fits in its space
    if( (prev ? (prev->m_totalbytes - prev->m_databegin + prev->m_datasize) : 0) > m_totalbytes - m_databegin ||
        m_datasize - m_databegin > m_spacesize )
    {
      #ifdef DEBUG
      std::cout << "corruption: invalid databegin -" << m_databegin << ", data " << m_datasize << std::endl;
      #endif
      
      m_datasize = 0; //Discard frame's data
      m_corruption = (m_datasize != 0); //If no data were discarded, everything is fine
    }
    
    //Read data, if we'll need them
    if( m_stream->WriteMode() == nowrite )
      bs.Seek(8 * m_fh->FrameSize() - headersize);
    else
      m_data = new AudioData(m_spacesize, m_datasize - m_databegin, bs);
    
    return true;
  }    
  catch(eEOF e) //End of file
  {
    if(e.type() == eEOF::final) //End of last file: stop processing
    {  
      Cleanup();
      throw e;
    }
    
    #ifdef DEBUG
    std::cout << "##########################################\n";
    #endif
    
    m_corruption = false;
    m_firstframe = true;
    return false;
  }
}
    
MP3Frame *MP3Frame::Destroy()
{
  #ifdef DEBUG
  std::cout << "destroy @" << m_totalbytes << std::endl;
  #endif
  
  MP3Frame *frame = m_next; //### There must be a local variable!
  delete this;
  return frame;
}    

MP3Frame *MP3Frame::Next()
{
  if(!m_next) //Read a new frame if there isn't one
    m_next = new MP3Frame(m_stream, this);
  
  return m_next;
}

void MP3Frame::WriteHeaders(Bitstream &bs)
{
  //Bitstream::Write() writs high-order bits first
  unsigned int sync = mp3::sync << (32 - mp3::syncbits);
  bs.Write(&sync, mp3::syncbits); //Write synchronization bits
  
  if(m_fh) m_fh->Write(bs); //Frame Header
  if(m_fh->isCRC()) bs.Write(&m_CRC, mp3::CRCbits); //CRC
  if(m_dh) m_dh->Write(bs); //Data Header
  
  //Channel Headers
  for(int i = 0; i < 2; i++)
    for(int j = 0; j < m_fh->NumChannels(); j++)
      if(m_ch[i][j]) m_ch[i][j]->Write(bs);
}

void MP3Frame::Write(Bitstream &bs)
{
  #ifdef DEBUG
  std::cout << "-----------------" << std::endl;
  std::cout << "writeframe @" << m_totalbytes << ", size " << m_spacesize << ", data " << m_datasize << " begin -" << m_databegin << std::endl;
  #endif
  
  assert(m_stream->WriteMode() != nowrite);

  //Fastwrite: simply write headers and data
  if(m_stream->WriteMode() == fastwrite)
  {
    WriteHeaders(bs);
    if(m_data) m_data->Write(bs);
  }
  else
  {  
    //If the data won't fit in frame
    if(m_datasize - m_newdatabegin > m_spacesize)
      Enlarge(m_datasize - m_newdatabegin - m_spacesize);
      
    m_dh->SetDataBegin(m_newdatabegin); //Save newdatabegin in Data Header
    WriteHeaders(bs); //Must be after SetDataBegin!
      
    m_locked = true; //Lock the frame so it won't be deleted
    //Write data from stream, use new (possibly enlarged) size of frame,
    // also position in frame must be corrected (so the end would match next frame)
    m_stream->WriteData(bs, m_spacesize + m_sizechange, m_totalbytes - m_sizechange);
    m_locked = false;
  }
}
    
void MP3Frame::SetDataBegin(int startpos)
{
  int maxdatabegin = m_dh->MaxDataBegin();
  
  //Offset between position of frame space and its data
  m_newdatabegin = m_totalbytes - startpos; 
  
  //If there's too little data, add padding bytes
  if(m_newdatabegin > maxdatabegin)
  {  
    m_stuffing = m_newdatabegin - maxdatabegin;
    m_newdatabegin = maxdatabegin;
  }
}

void MP3Frame::Enlarge(int num)  
{
  //Try to resize at least by num bytes, -1 is invalid bitrate
  while(m_fh->Bitrate() != -1 && m_sizechange < num)
  {  
    m_fh->Enlarge();
    m_sizechange = m_fh->FrameSize() - m_framesize;
  }
  
  if(m_fh->Bitrate() == -1) //This probably would never happen
    throw eCannotProceed("Cannot resize frame, it already has maximum bitrate");
  
  #ifdef DEBUG
  std::cout << "resized @" << m_totalbytes << " to " << m_sizechange + m_spacesize << std::endl;
  #endif
}

void MP3Frame::WriteData(Bitstream &bs, MP3Frame *ff, int cutstart, int cutend)
{
  //First byte of frame data in stream
  int start = m_totalbytes - m_databegin - m_stuffing + cutstart; 
  //End of frame data in stream
  int end = m_totalbytes - m_databegin + m_datasize - cutend; 
  int stuffing = 0; //How many padding bytes should be written
  
  #ifdef DEBUG
  std::cout << "write @" << m_totalbytes << " of size " << m_datasize << '(' << DataSize() << ") cut " << cutstart << ' ' << cutend << std::endl;
  #endif
  
  assert(cutstart >= 0);
  assert(cutend >= 0);
  assert(start < end);
  
  if(cutstart < m_stuffing) //Some padding bytes need to be written
  {
    if(end - start < m_stuffing - cutstart) 
      stuffing = end - start; //All bytes are padding
    else
      stuffing = m_stuffing - cutstart; //Some of bytes are padding
    
    bs.WriteNullBytes(stuffing); //Write pading (zero bytes)
    start += m_stuffing - cutstart; //Continue with real data
    
    #ifdef DEBUG
    std::cout << "written stuffing " << stuffing << std::endl;
    #endif
  }
  
  assert(start >= ff->m_totalbytes);
  
  while(true) //Welcome to the hell. This is the ugliest part of this program.
  {
    //If there is something to write and the data are in ff
    if(start < end && ff->m_totalbytes + ff->m_spacesize > start)
    {
      //Position of first byte to write in frame's data (not stream)
      int startbyte = (start > ff->m_totalbytes) ? (start - ff->m_totalbytes) : 0;
      //End of block to write from frame's data
      int endbyte = (end < ff->m_totalbytes + ff->m_spacesize) ? (end - ff->m_totalbytes) : ff->m_spacesize;
      
      if(ff->m_data) ff->m_data->Write(bs, startbyte, endbyte); //Finally write the real data
      
      #ifdef DEBUG
      std::cout << "written bytes " << startbyte << '-' << endbyte << " from frame @" << ff->m_totalbytes << std::endl;
      #endif
    }
    
    //There MUST be '<' (otherwise it can destroy the frame being written)
    if(ff->m_next && ff->m_next->m_totalbytes < end)  
      ff = ff->m_next; //If some data are in next frame, continue. 
    else
    {  
      m_stream->SeekTo(ff); //Delete "emptied" frames
      return;
    }
  }
}

void MP3Frame::SetFirst()
{
  m_stream->SetFirst(this);
}

bool MP3Frame::isLocked() const
{
  return m_locked;
}

bool MP3Frame::CorruptionFound() const
{
  return m_corruption;
}

int MP3Frame::Bitrate() const
{
  return m_fh->Bitrate();
}

int MP3Frame::DataSize() const
{  
  return m_datasize + m_stuffing; //Return size with padding data 
}                                 // (ie. size in output stream)

void MP3Frame::ChangeGain(int n)
{
  //Change volume in every granule and channel
  for(int i = 0; i < 2; i++)
    for(int j = 0; j < m_fh->NumChannels(); j++)
      if(m_ch[i][j]) m_ch[i][j]->ChangeGain(n);
}

bool MP3Frame::isSilent() const
{
  int size = 0;
  
  //HACK: Just check if both channels of second granule
  //      have no data, works better than it seems
  for(int j = 0; j < m_fh->NumChannels(); j++)
    size += m_ch[1][j]->DataSize();
  
  return size == 0;
}
    
void MP3Frame::ShowInfo(std::ostream &str) const
{
  static const char *versiontext[] = { 0, "MPEG1", "MPEG2", "MPEG2.5" };
  static const char *modetext[] = { "Stereo", "Joint Stereo", "Dual Channel", "Single Channel" };
    
  if( !m_fh->isValid() )
    str << "Invalid file!" << std::endl;
  else
  {  
    str << "Format: " << versiontext[m_fh->Version()] << " layer " << m_fh->Layer() << std::endl;
    str << "Bitrate: " << m_fh->Bitrate() << " kbps" << std::endl;
    str << "Frequency: " << m_fh->Frequency() << " kHz" << std::endl;
    str << "Mode: " << modetext[m_fh->Mode()];
    if(m_fh->isMSstereo()) str << " MS";
    if(m_fh->isIntensStereo()) str << " Intensity";
    str << std::endl;
  }
}
