/**
 * \file mp3core.cpp
 * @short Implementation of class MP3Core
 */

#include <iostream>

#include "mp3core.h"
#include "mp3frame.h"
#include "mp3iterator.h"
#include "bitstream.h"
#include "except.h"
#include "conv.h"

MP3Core::MP3Core()
{
  m_outfile = 0;
  m_infiles = 0;
  m_showinfo = false;
  m_gain = 0;  
  m_cutstart = m_cutend = 0;
  m_autocutstart = false;
  m_autocutend = false;
  m_numframes = 0;
}

bool MP3Core::Params(int argc, const char *argv[])
{
  int i;
  
  //Skip last argument, it must be a filename
  for(i = 1; i < argc - 1; i++)
  {
    if(argv[i][0] == '-' && argv[i][1]) 
    { //Arguments are one-character only, beginning with one dash
      switch(argv[i][1])
      {
        case 'h': //Show usage message
          return false;
        case 'o': //Output file
          m_outfile = argv[++i];
          break;
        case 'g': //Gain (need to be converted from dB to numbers used in MP3)
          m_gain = Conv::dB2Gain(argv[++i]);
          break;
        case 'v': //Show various information
          m_showinfo = true;
          break;
        case 'c': //Cut frames
          switch(argv[i][2])
          {
            case 's': //Cut from beginning
              if(*argv[++i] == 'a') //Autodetect silence
                m_autocutstart = true;
              else //Convert time to frame number
                m_cutstart = Conv::Time2Frame(argv[i]);
              break;
            case 'e': //Cut from end
              if(*argv[++i] == 'a') //Autodetect silence
                m_autocutend = true;
              else //Convert time to frame number
                m_cutend = Conv::Time2Frame(argv[i]);;
              break;
          }
          break;
        case 't': //Edit tag
          i += m_tag.ParseParam(argv[i][2], argv[i+1]);
          break;
      }
    }
    else
      break; //If this argument is not a parameter, all others (including this one) are input files
  }
  
  m_infiles = argv + i; //HACK: Input files are taken as they are (null terminated array of char*)
  return argc > i; //At least one input file is needed
}

void MP3Core::GetInfo()
{
  bool start = m_autocutstart; //Whether we should detect silence at beginning
  int numloud = 0; //Number of continuous non-silent frames
  int avgbitrate = 0; //Average bitrate
  
  try {
    Bitstream bs(m_infiles, Bitstream::in); //Construct a Bitstream from all input files
    MP3Iterator frame(bs, MP3Frame::nowrite); //Iterator of all frames, write is turned off
    
    if(m_showinfo) //Show info from ID3 tags, bitrate, etc.
    {
      m_tag.Show(std::cout); //Show ID3 tag information
      frame->ShowInfo(std::cout); //Show bitrate, version, etc. of first frame
    }
    
    while(true) //Real processing
    {
      //Log corrupted frames (done here only if no write will be done)
      if( !m_outfile && frame->CorruptionFound() ) 
        std::cerr << "Corruption found at " << Conv::Frame2Time(m_numframes) << "!" << std::endl;
      
      avgbitrate += frame->Bitrate(); //Sum of bitrate (not yet average)
      m_numframes++; //Count frames
      
      if(m_autocutstart && start) //Detect silent frames at the beginning 
      {                           // (if requested and we're still at the beginning)
        if( frame->isSilent() )
        {  
          m_cutstart = m_numframes; //All frames since beginning are silent
          numloud = 0;
        }
        else if(++numloud > mp3::maxloud) //If there are more than mp3::maxloud consecutive
          start = false;                  // non-silent frames, the silent block has ended
      }
      
      if(m_autocutend && !start) //Detect silent frames at the end
      {
        if( frame->isSilent() ) //Silent frame ends a block of non-silent ones
          numloud = 0;
        else if(++numloud > mp3::maxloud) //Block of more than mp3::maxloud non-silent frames
          m_cutend = m_numframes;         // makes the detection began from zero (current frame)
      }
      
      ++frame;
    }
  }
  catch(eEOF) //End of last input file
  {
    avgbitrate /= m_numframes; //Sum -> average
    
    if(m_autocutend) //Convert frame number to count of frames till the end of file
      m_cutend = m_numframes - m_cutend;
    
    if(m_showinfo) //Display file duration, bitrate and detected silence
    {  
      std::cout << "Duration: " << Conv::Frame2Time(m_numframes) << std::endl;
      std::cout << "Average bitrate: " << avgbitrate << " kbps" << std::endl;
      
      if(m_autocutstart || m_autocutend)
        std::cout << "Silence: " << Conv::Frame2Time(m_cutstart) << " at the beginning and " 
                                  << Conv::Frame2Time(m_cutend) << " at the end." << std::endl;
    }
  }
}

bool MP3Core::WriteResult()
{
  int j = 0; //Number of current frame
  
  try {
    Bitstream bs(m_infiles, Bitstream::in); //Open input file(s)
    Bitstream bs_out(m_outfile, Bitstream::out); //Open output file
    
    //Get first frames, frames can be written in "fast" mode when there's no need do move data
    // (when no frames are removed from beginning and we're processing just one input file)
    MP3Iterator frame(bs, (m_cutstart == 0) ? MP3Frame::fastwrite : MP3Frame::normal);
    
    while(true) //Real processing
    {
      if( frame->CorruptionFound() ) //Log corrupted frames 
        std::cerr << "Corruption found at " << Conv::Frame2Time(j) << "!" << std::endl;
      
      if(m_gain) frame->ChangeGain(m_gain); //Change frame volume
      
      if(m_cutend && j >= m_numframes - m_cutend) //Immediately exit when no more
        break;                                    // frames should be written
      
      //### When this frame should be first, it must be set first in stream, otherwise
      //### data would be written from really first frame and things would break 
      if(j == m_cutstart)    
        frame->SetFirst(); 
      
      if(j >= m_cutstart) //Finally write the frame
        try {
          frame->Write(bs_out);
        } 
        catch(eCannotProceed e) //When frames needs to be enlarged
        {                       // but it already has maximum size
          std::cerr << "Sorry, this operation is not possible: " << e.what() << std::endl;
          return false;
        }
      
      ++frame; //Next frame (iterator)
      j++;
    }
  } 
  catch(eEOF) { } //EOF is normal end of processing loop
  
  m_tag.Write(m_outfile);
  return true;    
}

bool MP3Core::Process()
{
  try {
    m_tag.Read(*m_infiles); //Extract tags
    
    //Process files (without writing) if needed
    if(m_showinfo || m_cutend || m_autocutstart || m_autocutend || !m_outfile)
      GetInfo();
    
    if(m_outfile) //Write result if output file was given
      return WriteResult();
  }
  catch(eOpenFailed e)
  {
    std::cerr << "Can't open file `" << e.filename() << "'!" << std::endl;
    return false;
  }

  return true; //Just to please compiler
}

void MP3Core::ShowHelp(const char *filename)
{
  std::cout << "Tool for various lossless operations on MP3 files." << std::endl; 
  std::cout << "Usage: " << filename << " [parameters] input files" << std::endl;
  std::cout << "Parameters: " << std::endl;
  std::cout << "  -o file.mp3  Output file (input files are never changed!) " << std::endl;
  std::cout << "  -g N         Change volume by N dB " << std::endl;
  std::cout << "  -v           Show file duration, bitrate, tag information etc. " << std::endl;
  std::cout << "  -cs time     Cut specified time from beginning of file, " << std::endl;
  std::cout << "               if time is 'a', digital silence is removed " << std::endl;
  std::cout << "  -cs time     Cut specified time from end of file, " << std::endl;
  std::cout << "               if time is 'a', digital silence is removed " << std::endl;
  std::cout << "  -t1          Keep ID3v1 tag (or create when some field is set) " << std::endl;
  std::cout << "  -t2          Keep ID3v1 tag (or create when some field is set) " << std::endl;
  std::cout << "  -tx          Don't read ID3 tags, create from scratch " << std::endl;
  std::cout << "  -tt value    Set Title field in ID3 tag " << std::endl;
  std::cout << "  -ta value    Set Artist field in ID3 tag " << std::endl;
  std::cout << "  -tA value    Set Album field in ID3 tag " << std::endl;
  std::cout << "  -tn value    Set Track Number in ID3 tag " << std::endl;
  std::cout << "  -ty value    Set Year field in ID3 tag " << std::endl;
  std::cout << "  -tc value    Set Comment field in ID3 tag " << std::endl;
  std::cout << "  -tg value    Set Genre field in ID3 tag " << std::endl;
  std::cout << "               (only standard ID3 genres can be used) " << std::endl;
}
