/**
 * \file id3tag.h
 * @short Declaration of class ID3Tag
 */

#ifndef _ID3TAG_H_
#define _ID3TAG_H_

#include <id3v2tag.h> //Taglib

/** 
 * @short Operations with ID3 tags using Taglib
 *
 * Used by MP3Core, reads and writes ID3 tags and changes 
 * them as requested by command-line parameters.
 * @warning Input and output encoding is UTF-8!
 */
class ID3Tag
{
  public:  
    /**
     * Constructor, only initializes everything to zero
     */
    ID3Tag();
    
    /**
     * Destructor, deletes ID3 tags
     */
    ~ID3Tag();
    
    /**
     * Processes tag-related parameters
     * @param param Sub-parameter of -t
     * @param value Next parameter
     * @return Whether next parameter (value) was used
     */
    int ParseParam(const char param, const char *value);
    
    /**
     * Extracts ID3 tag from input file using Taglib
     * @warning It gets a filename, not a Bitstream, because of Taglib
     * @param filename Name of file to extract tags from
     */
    void Read(const char *filename);
    
    /**
     * Writes ID3 tag to output file (extracted tags from input file + changes)
     * @warning It gets a filename, not a Bitstream, because of Taglib
     * @param filename Name of file to write tags to
     */
    void Write(const char *filename) const;
    
    /**
     * Show information from ID3 tag as text (UTF-8)
     * @param ostr Output stream
     */
    void Show(std::ostream &ostr) const;

  private:
    bool m_usetagv1; //Whether to write ID3v1 tag
    bool m_usetagv2; //Whether to write ID3v2 tag
    bool m_noreadtags; //Don't read ID3 tags, create them from scratch
    TagLib::Tag *m_tagv1, *m_tagv2; //Tags from input file
    TagLib::ID3v2::Tag *m_newtag; //Tag constructed from parameters
};

#endif //_ID3TAG_H_
