/**
 * \file id3tag.cpp
 * @short Implementation of class ID3Tag
 */

#include <cassert>
#include <ostream>

//Taglib
#include <mpegfile.h>
#include <id3v1tag.h>
      
#include "id3tag.h"
      
ID3Tag::ID3Tag()
{
  //UTF-8 encoding for text fields int ID3v2 tag
  TagLib::ID3v2::FrameFactory::instance()->setDefaultTextEncoding(TagLib::String::UTF8);
  
  m_usetagv1 = m_usetagv2 = false;
  m_noreadtags = false;
  m_tagv1 = m_tagv2 = 0;
  m_newtag = new TagLib::ID3v2::Tag;
}
    
ID3Tag::~ID3Tag()
{
  delete m_tagv1;
  delete m_tagv2;
  delete m_newtag;
}

int ID3Tag::ParseParam(const char param, const char *value)
{
  switch(param)
  {
    case '1': //Keep/create ID3v1 tag
      m_usetagv1 = true;
      return 0;
    case '2': //Keep/create ID3v2 tag
      m_usetagv2 = true;
      return 0;
    case 'x': //Don't read ID3 tags
      m_noreadtags = true;
      return 0;
    case 't': //Title
      m_newtag->setTitle(TagLib::String(value, TagLib::String::UTF8));
      return 1;
    case 'a': //Artist
      m_newtag->setArtist(TagLib::String(value, TagLib::String::UTF8));
      return 1;
    case 'A': //Album
      m_newtag->setAlbum(TagLib::String(value, TagLib::String::UTF8));
      return 1;
    case 'n': //Track number
      m_newtag->setTrack(atoi(value));
      return 1;
    case 'g': //Genre
      m_newtag->setGenre(TagLib::String(value, TagLib::String::UTF8));
      return 1;
    case 'y': //Year
      m_newtag->setYear(atoi(value));
      return 1;
    case 'c': //Comment
      m_newtag->setComment(TagLib::String(value, TagLib::String::UTF8));
      return 1;
    default:
      return 0;
  }
}

void ID3Tag::Read(const char *filename)
{ 
  assert(filename);
  
  if(m_noreadtags || (!m_usetagv1 && !m_usetagv2))
    return;
    
  TagLib::MPEG::File f(filename); //Open file
  
  if(m_usetagv1) //Read ID3v1 tag
  { 
    m_tagv1 = new TagLib::ID3v1::Tag;
    //### Must be copied, otherwise it will be destroyed with f!
    TagLib::Tag::duplicate(f.ID3v1Tag(true), m_tagv1);
  }  
  
  if(m_usetagv2) //Read ID3v2 tag
  { //HACK: ID3v2 returns genre as number after duplicate()
    m_tagv2 = new TagLib::ID3v1::Tag; 
    //### Must be copied, otherwise it will be destroyed with f!
    TagLib::Tag::duplicate(f.ID3v2Tag(true), m_tagv2);
  }  
}

void ID3Tag::Write(const char *filename) const
{
  assert(filename);
  TagLib::MPEG::File f(filename); //Open file

  if(m_usetagv1) //Write ID3v1 tag
  {
    TagLib::Tag *tag = f.ID3v1Tag(true);
    TagLib::Tag::duplicate(m_newtag, tag);
    
    if(m_tagv1) 
      TagLib::Tag::duplicate(m_tagv1, tag, false);
  }
  
  if(m_usetagv2) //Write ID3v2 tag
  {
    TagLib::Tag *tag = f.ID3v2Tag(true);
    TagLib::Tag::duplicate(m_newtag, tag);
    
    if(m_tagv2) 
      TagLib::Tag::duplicate(m_tagv2, tag, false);
  }

  //Save selected tags, strip others
  f.save((m_usetagv1 ? TagLib::MPEG::File::ID3v1 : 0) | 
         (m_usetagv2 ? TagLib::MPEG::File::ID3v2 : 0), true);
}

void ID3Tag::Show(std::ostream &ostr) const
{          
  TagLib::Tag *tag = 0;
  
  if(m_tagv1 && !m_tagv1->isEmpty())
    tag = m_tagv1;
  
  if(m_tagv2 && !m_tagv2->isEmpty())
    tag = m_tagv2;
  
  if(tag)
  {
    ostr << "Artist: "  << tag->artist().to8Bit(true)  << std::endl;
    ostr << "Title: "   << tag->title().to8Bit(true)   << std::endl;
    ostr << "Album: "   << tag->album().to8Bit(true)   << std::endl;
    ostr << "Track: "   << tag->track()                << std::endl;
    ostr << "Genre: "   << tag->genre().to8Bit(true)   << std::endl;
    ostr << "Year: "    << tag->year()                 << std::endl;
    ostr << "Comment: " << tag->comment().to8Bit(true) << std::endl;
    ostr << std::endl;
  }
}
