/**
 * \file bitstream.cpp
 * @short Implementation of class Bitstream
 */

#include <cassert>

#define TAGV1
#include "mp3_format.h"
#include "bitstream.h"
#include "except.h"

#ifdef DEBUG
#include <iostream>
#endif

Bitstream::Bitstream(const char *filename, int mode) : m_stream()
{
  m_leftbits = 0;
  m_char = 0;
  m_mode = mode;
  m_files = &m_NULL; //No next file
  
  assert(filename);
  Open(filename);
}

Bitstream::Bitstream(const char **files, int mode) : m_stream()
{
  assert(files);
  m_leftbits = 0;
  m_char = 0;
  m_mode = mode;
  m_files = files + 1; //Store rest of files
  
  assert(*files);
  Open(*files); //Open first file
}

void Bitstream::Read(unsigned int *data, int numbits)
{
  int havebits = 32; //Go to next (first) int
  int i = -1; //First operation is i++

  assert(data);
  assert(numbits > 0);
  
  while(numbits > 0) //While we need to read some more bits
  {  
    if(havebits == 32) //Int is full
    {
      data[++i] = m_char << 24; //Place the char to high-order bits
      numbits -= m_leftbits; //We have as much bits 
      havebits = m_leftbits; // as in m_char
      m_char = 0;
      m_leftbits = 0;
    }
    else
    {
      //This is ugly, but m_char muset be unsigned for shifts
      ReadBytes(reinterpret_cast<char*>(&m_char), 1);
      
      if(havebits <= 24) //We can read a full character
      {
        data[i] |= m_char << (24 - havebits); //Place the char after read bits
        havebits += 8;
        numbits -= 8;
        m_char = 0;
      }
      else //The character is on integer boundary
      {
        m_leftbits = havebits - 24; //Some bytes have to be kept in char
        data[i] |= m_char >> m_leftbits; //Other bits are appended to int
        m_char <<= 32 - havebits; //Remove appended bits 
        numbits -= 32 - havebits;
        havebits = 32; //Integer is full, move to next in next step
      }
    }
  }
  
  if(numbits < 0) //We have more bits than we need
  {
    m_char >>= -numbits; //Make place for these bits
    m_char |= data[i] << (havebits + numbits) >> 24; //Move the bits back
    m_leftbits -= numbits; //And remember how many are they
  }
}

void Bitstream::Write(const unsigned int *data, int numbits)
{
  int i = 0;
  int havebits = (numbits < 32) ? numbits : 32; //Number of bits in first int
  unsigned int dataword = data[0]; //Copy bits (don't change input)
  
  while(numbits > 0) //While there's something to write
  {
    if(havebits + m_leftbits >= 8) //We have at least one byte
    {
      m_char |= dataword >> (24 + m_leftbits); //Complete the char with bits from dataword
      WriteBytes(reinterpret_cast<char*>(&m_char), 1);
      dataword <<= 8 - m_leftbits; //Remove written bits
      havebits -= 8 - m_leftbits; 
      numbits -= 8 - m_leftbits;
      m_leftbits = 0; 
      m_char = 0;
    }
    else //We don't have a full byte, we have to keep the bits
    {
      m_char |= dataword >> (24 + m_leftbits); //Append the bits to m_char
      m_leftbits += havebits;
      numbits -= havebits;
      havebits = (numbits < 32) ? numbits : 32; //Continue to next integer
      if(numbits > 0) dataword = data[++i]; //Check needed to prevent possible segfault
    }    
  }
}

void Bitstream::Seek(int numbits)
{
  assert(m_mode == in);
  assert(numbits > 0);
  
  if(numbits <= m_leftbits) //Just a little seek in m_char
  {                         // (drop a few bits)
    m_leftbits -= numbits;
    m_char <<= numbits;
  }
  else
  {
    numbits -= m_leftbits; //Throw bits in m_char
    m_stream.seekg(numbits >> 3, std::ios::cur); //Real seek in stream
    numbits &= 7; //Left bits (modulo 8)
    
    if(numbits)
    {   //Read character so we can seek in it
        ReadBytes(reinterpret_cast<char*>(&m_char), 1);
        m_char <<= numbits; //Seek in m_char
        m_leftbits = 8 - numbits;
    }  
    else
     m_leftbits = 0;
  }
}

int Bitstream::Find(unsigned int pattern, int numbits)
{
  int aligned = (numbits + 7) & ~7; //Align pattern to byte boundary
  unsigned int mask = ~0 << (32 - numbits); //Mask to select bits in pattern
  unsigned int buffer = 0; //Read bytes to compare with pattern
  int skipped = 0; //Bytes read before match was found
  
  m_leftbits = 0; //Match is always on byte boundary
  pattern <<= 32 - numbits; //Move the pattern to high-order bits
  
  Read(&buffer, aligned); //Reads enough bytes to try a first comparison
  while( (buffer & mask) != pattern ) //Until pattern is found
  {  
    ReadBytes(reinterpret_cast<char*>(&m_char), 1);
    skipped++;
    buffer <<= 8; //Add character to buffer
    buffer |= m_char << (32 - aligned);
  }  
  
  m_char = buffer >> (24 - numbits); //Remove bits left from pattern
  m_leftbits = aligned - numbits;

  return skipped;
}

void Bitstream::ReadBytes(char *bytes, int num)
{
  int newnum = num;
  assert(m_mode == in);
  
  if(num + m_stream.tellg() >= m_size)
    newnum = m_size - m_stream.tellg();
  
  if(!m_stream.read(bytes, newnum) || newnum < num) //EOF
  {  
    newnum = m_stream.gcount(); //Number of bytes really read
    
    #ifdef DEBUG
    std::cout << "read " << newnum << " bytes instead of " << num << "!" << std::endl;
    #endif
    
    if(*m_files) //There's some next file
    {  
      m_stream.close();
      m_stream.clear(); //Must be here for some reason!
      m_leftbits = 0;
      
      Open(*m_files);
      m_files++;
      throw eEOF(eEOF::partial, newnum); //Report state by exception
    }
    else
      throw eEOF(eEOF::final, newnum); //Total EOF
  }
}

void Bitstream::WriteBytes(const char *bytes, int num)
{
  assert(m_mode == out);
  m_stream.write(bytes, num);
}

void Bitstream::WriteNullBytes(int num)
{
  assert(m_mode == out);
  for(int i = 0; i < num; i++)
    m_stream.put(0);
}

void Bitstream::Open(const char *filename)
{
  switch(m_mode)
  {
    case in:  m_stream.open(filename, std::fstream::in  | std::fstream::binary); break;
    case out: m_stream.open(filename, std::fstream::out | std::fstream::binary); break; 
  }

  if(!m_stream) //File not found, send filename in what()
    throw eOpenFailed(filename);

  if(m_mode == in) //Get file size (without ID3v1 tag)
  {
    m_stream.seekg(0, std::ios::end);
    m_size = m_stream.tellg();
    
    CheckTag();
    m_stream.seekg(0, std::ios::beg);
    
    #ifdef DEBUG
    std::cout << "File size " << m_size << " bytes" << std::endl;
    #endif
  }
}

void Bitstream::CheckTag()
{
  char tagheader[mp3_tag_v1::headersize];
  
  if(m_size > mp3_tag_v1::size)
  {  
    m_stream.seekg(-mp3_tag_v1::size, std::ios::end);
    
    if(m_stream.read(tagheader, sizeof(tagheader)) && !memcmp(tagheader, mp3_tag_v1::header, sizeof(tagheader)))
    {  
      #ifdef DEBUG
      std::cout << "ID3v1 tag detected" << std::endl;
      #endif
      
      m_size -= mp3_tag_v1::size;
    }
  }
}

const char *Bitstream::m_NULL = 0;
