/**
 * \file bitarray.h
 * @short Implementation of template Bitarray
 */

#ifndef _BITARRAY_H_
#define _BITARRAY_H_

#include "bitstream.h"

/**
 * @short Compact struct of items with variable bit length
 * 
 * In ordinary C struct items must end on byte boundary, 
 * but in MP3 files fields have variable bit length. This 
 * class works line C struct, but unfortunately operator '.'
 * can't be overloaded, I had to use '^' with low priority. 
 * Items can be selected via an enum to emulate C struct.
 */
template<typename T> class Bitarray 
{
  public:
    
    /**
     * Constructor
     *
     * Reads needed bits from Bitstream
     *
     * @param bs Input Bitstream
     */
    Bitarray(Bitstream &bs)
    {
      bs.Read(m_data, T::size);
    }
    
    /**
     * Writes bits to Bitstream
     *
     * @param bs Output Bitstream
     */
    void Write(Bitstream &bs) const
    {
      bs.Write(m_data, T::size);
    }
    
    /**
     * Returns value of an field 
     *
     * @warning Field can't spread over integer boundary!
     * @param i Field number
     * @return Field value
     */
    int Get(int i) const
    {
      int startbit = T::bits[i] & 0x1F; //Start bit position in integer
      int endbit = T::bits[i+1] & 0x1F; //End bit position in integer
      
      //Get integer which contains the field and remove bits of neighbour fields
      return m_data[T::bits[i] >> 5] << startbit >> (32 - endbit + startbit);
    }

    /**
     * Sets value of a field
     *
     * @warning Field can't spread over a byte boundary!
     * @param i Field number
     * @param x Field value
     */
    void Set(int i, int x)
    {
      int startbit = T::bits[i] & 0x1F; //Start bit position in integer
      int endbit = T::bits[i+1] & 0x1F; //End bit position in integer     
      int intnum = T::bits[i] >> 5; //Integer which contains the field
      //Set bits belonging to field to 1 and other bits to 0
      unsigned int mask = ((unsigned int)(~0) >> startbit) &
                          ((unsigned int)(~0) << (32 - endbit));
      
      //Make field bits zero and then set them to new value
      m_data[intnum] &= ~mask;
      m_data[intnum] |= (x << (32 - endbit)) & mask;
    }

    /**
     * Get maximum value of a field
     *
     * @param i Field number
     * @return Maximum value of a field
     */
    int Max(int i) const
    {
      return (1 << (T::bits[i+1] - T::bits[i])) - 1;
    }
    
    /**
     * Returns value of a field
     *
     * Just a shortcut for get() to imitate C struct, so 
     * bitarray^field can be used instead of bitarray.get(field), 
     * if 'field' is enumeration value.
     *
     * @warning Operator^ has a low priority!
     * @param i Field number
     * @return Field value
     */
    unsigned int operator^ (int i) const
    {
      return Get(i);
    }
      
  private:
    unsigned int m_data[((T::size) + 31) >> 5]; //Data, minimal size to fit all bits

};

#endif //_BITARRAY_H_
