%%%%%%%%%% GRAPH STRUCTURE %%%%%%%%%%
nodenum(  node(V,_,_,_,_,_,_),V). % Cislo vrcholu
edges(    node(_,E,_,_,_,_,_),E). % Seznam sousedu 
pair(     node(_,_,P,_,_,_,_),P). % S cim je vrchol sparovan
mark(     node(_,_,_,M,_,_,_),M). % Oznaceni navstiveneho vrcholu
clone(    node(_,_,_,_,C,_,_),C). % Klon (pro kopirovani)
contr(    node(_,_,_,_,_,O,_),O). % Kam je vrchol zkontrahovan
realcontr(node(_,_,_,_,_,_,R),R). % "Zkratka": kam je _doopravdy_ zkontrahovan

%%%%%%%%%% GRAPH FUNCTIONS %%%%%%%%%%

% Zda ma vrchol Node znacku X
marked(Node,X) :- mark(Node,M), nonvar(M), M == X.
notmarked(Node,X) :- mark(Node,M), ( var(M); M \= X ).

% Zda je vrchol Node zkontrahovan
contracted(Node) :- contr(Node,Contr), nonvar(Contr).
notcontr(Node) :- contr(Node,Contr), var(Contr).

% Zda je vrchol Node sparovan
unpaired(Node) :- pair(Node,Pair), var(Pair).
paired(Node) :- pair(Node,Pair), nonvar(Pair).

%%%%%%%%%% GRAPH UTILITIES %%%%%%%%%%

% Sparovani vrcholu V1 a V1 (musi se provest v klonech, hodnoty nelze menit)
pair_nodes(V1,V2) :- clone(V1,C1), clone(V2,C2), pair(C1,C2), pair(C2,C1).

% Pokud je Node kontrahovan, je RealNode vrchol do nehoz je zkontrahovan, jinak RealNode = Node
realnode(Node,Node) :- notcontr(Node).
realnode(Node,RealNode) :- contracted(Node), realcontr(Node,RealNode).

%%%%%%%%%% COPY %%%%%%%%%%

% Zkopiruje graf Graph, vrati kopii Copy
copy_graph(Graph,Copy) :- graph_copy(Graph,Copy), calc_realcontr(Copy).

% Zkopiruje graf (vytvori seznam klonu vrcholu a zaroven vytvari tyto klony)
graph_copy([Node|List],[NewNode|NewList]) :- clone(Node,NewNode),  
  nodenum(Node,V), nodenum(NewNode,V),
  edges(Node,Edges), edges(NewNode,NewEdges), edges_copy(Edges,NewEdges), 
  ( paired(NewNode); unpaired(Node); ( pair(Node,Pair), clone(Pair,NewPair), pair(NewNode,NewPair) ) ),
  ( notcontr(Node); ( contr(Node,Contr), clone(Contr,ContrClone), contr(NewNode,ContrClone) ) ),
  graph_copy(List,NewList).
graph_copy([],[]).

% Zkopiruje hrany z vrcholu (vytvori seznam klonu sousedu)
edges_copy([Node|List],[NewNode|NewList]) :- clone(Node,NewNode), edges_copy(List,NewList).
edges_copy([],[]).

% Spocita, kam jsou vrcholy zkontrahovane ("zkratky")
calc_realcontr([Node|List]) :- notcontr(Node), calc_realcontr(List).
calc_realcontr([Node|List]) :- contr(Node,Contr), notcontr(Contr), realcontr(Node,Contr), calc_realcontr(List).
calc_realcontr([Node|List]) :- contr(Node,Contr), contracted(Contr), realcontr(Contr,RealContr), realcontr(Node,RealContr),
  calc_realcontr(List).
calc_realcontr([]).

%%%%%%%%%% AUGMENTING PATH %%%%%%%%%%

% Hledani volne stridave cesty, zaroven zameni sparovane a nesparovane hrany na nalezene ceste
augmenting_path([Node|List]) :- unpaired(Node), notcontr(Node), dfs_path(Node,Result), !, ( Result == true; augmenting_path(List) ).
augmenting_path([_|List]) :- augmenting_path(List).

% Prochazeni do hloubky z vrcholu Node1, hledani VSC, vraci v Result true/false jestli cestu naslo -
dfs_path(Node1,false) :- realnode(Node1,Node), marked(Node,1).
dfs_path(Node1,Result) :- realnode(Node1,Node), notmarked(Node,1), mark(Node,1), edges(Node,Edges), dfs_path_edges(Node,Edges,Result).

% Licha vrstva: jdeme po parovaci hrane (pokud neni, nasli jsme VSC)
dfs_path_pair(Node1,false) :- realnode(Node1,Node), marked(Node,1).
dfs_path_pair(Node1,true) :- realnode(Node1,Node), notmarked(Node,1), unpaired(Node).
dfs_path_pair(Node1,Result) :- realnode(Node1,Node), notmarked(Node,1), mark(Node,1), pair(Node,Pair), dfs_path(Pair,Result).

% Suda vrstva: prochazime vsechny sousedy, pokud do nektereho vede VSC, koncime s true, jinak zkousime dal
% parametry: vrchol Node, seznam jeho sousedu a navratova hodnota
dfs_path_edges(Node,[Edge|List],NewResult) :- dfs_path_pair(Edge,Result), 
  ( ( Result == true, NewResult = true, pair_nodes(Node,Edge) ); dfs_path_edges(Node,List,NewResult) ).
dfs_path_edges(_,[],false).

%%%%%%%%%% ODD CIRCLE %%%%%%%%%%

% Hledani liche kruznice v grafu, NewNode je vrchol, do ktereho se pripadne kruznice zkontrahuje
odd_circle([Node|List],NewNode) :- unpaired(Node), notcontr(Node), ( dfs_circle(Node,_,NewNode); odd_circle(List,NewNode) ).
odd_circle([_|List],NewNode) :- odd_circle(List,NewNode).

% Prochazeni do hloubky z vrcholu Node1, vraci koncovy bod kruznice NewEnd 
% (nebo nil pokud nejsme na kruznici), vrcholy se kontrahuji do NewNode
dfs_circle(Node1,NewEnd,NewNode) :- realnode(Node1,Node), notmarked(Node,1), mark(Node,1), edges(Node,Edges),
  dfs_circle_edges(Edges,End,NewNode), dfs_circle_mark(End,Node,NewNode),
  ( ( End == Node, NewEnd = nil, ( unpaired(End); ( pair(End,EndPair), pair_nodes(EndPair,NewNode) ) ) ); 
    ( End \= Node, NewEnd = End ) ).
dfs_circle(Node1,nil,[]) :- realnode(Node1,Node), marked(Node,1); marked(Node,2).

% Pokud je vrchol Node na kruznici, zkontrahuje se do NewNode
dfs_circle_mark(End,Node,NewNode) :- End \= nil, contr(Node,NewNode).
dfs_circle_mark(nil,_,_).

% Licha vrstva: pokud je vrchol v sude vrstve, mame lichou kruznici, jinak jdeme po parovaci hrane
dfs_circle_pair(Node1,Node,_) :- realnode(Node1,Node), marked(Node,1).
dfs_circle_pair(Node1,End,NewNode) :- realnode(Node1,Node), notmarked(Node,1), notmarked(Node,2), mark(Node,2), pair(Node,Pair), 
  dfs_circle(Pair,End,NewNode), dfs_circle_mark(End,Node,NewNode).

% Suda vrstva: prochazime vsechny sousedy, zkousime jestli nektery nevede do liche kruznice, jinak prochazime dal
% parametry: +List - seznam sousedu, -End - vrchol, kterým jsem do kružnice přišli, +NewNode - vrchol vzniklý kontrakcí kružnice
dfs_circle_edges([Edge|List],End,NewNode) :- dfs_circle_pair(Edge,End,NewNode); dfs_circle_edges(List,End,NewNode).

%%%%%%%%%% CONTRACT %%%%%%%%%%

% Zkontrahuje kruznici v grafu Graph, vysledek vrati v NewGraph
contract(Graph,NewGraph) :- length(Graph,N), nodenum(NewNode,N), odd_circle(Graph,NewNode), 
  contr_newnode_edges(Graph,NewNode,NewEdges), edges(NewNode,NewEdges), copy_graph([NewNode|Graph],NewGraph).

% Vyhleda sousedy noveho vrcholu NewNode (vznikleho kontrakci kruznice)
% contr_newnode_edges(+Graph,+NewNode,-Edges)
contr_newnode_edges([Node|List],NewNode,[Node|NewNodeEdges]) :- notcontr(Node), edges(Node,Edges), edge_to(Edges,NewNode), 
  contr_newnode_edges(List,NewNode,NewNodeEdges).
contr_newnode_edges([_|List],NewNode,NewNodeEdges) :- contr_newnode_edges(List,NewNode,NewNodeEdges).
contr_newnode_edges([],_,[]).

% edge_to(+Edges,+Node) - jestli je mezi hranami Edges nejaka hrana do vrcholu Node
edge_to([Node|_],Contr) :- contracted(Node), contr(Node,Contr).
edge_to([Node|_],Node).
edge_to([_|List],Node) :- edge_to(List,Node).

%%%%%%%%%% DECONTRACT %%%%%%%%%%

% Dekontrakce grafu, vysledek vrati jako NewGraph
decontract([NewNode|Graph],NewGraph) :- pair(NewNode,Pair), edges(Pair,Edges), edge_to_circle(Edges,NewNode,CircleNode),
  pair_nodes(Pair,CircleNode), circle_pair(CircleNode,NewNode), copy_graph(Graph,NewGraph).

% edge_to_circle(+Edges,+NewNode,-Node) - Najde mezi Edges vrchol na kruznici (zkontrahovany do NewNode), vrati ho v Node
edge_to_circle([Node|_],NewNode,Node) :- notmarked(Node,1), contracted(Node), contr(Node,NewNode).
edge_to_circle([Node|_],NewNode,Node1) :- notmarked(Node,1), contracted(Node), realcontr(Node,NewNode), contr(Node,Node1).
edge_to_circle([_|List],NewNode,Node) :- edge_to_circle(List,NewNode,Node).

% Sparuje dva vrchol Node na kruznici (tj. zkontrahovany do NewNode), pokud to lze - tim se zajisti stridave parovani
circle_pair(Node,NewNode) :- mark(Node,1), edges(Node,Edges), edge_to_circle(Edges,NewNode,NextNode),
  ( ( clone(Node,Clone), paired(Clone) ); pair_nodes(Node,NextNode) ), circle_pair(NextNode,NewNode).
circle_pair(_,_).

%%%%%%%%%% LOAD GRAPH %%%%%%%%%%

% Nacte graf Graph ze souboru Filename
load_graph(Filename,Graph) :- consult(Filename), graph(Graph), add_numbers(Graph,0).

% Ocislovani vrcholu grafu
add_numbers([Node|List],N) :- nodenum(Node,N), N1 is N + 1, add_numbers(List,N1).
add_numbers([],_).

%%%%%%%%%% MAIN %%%%%%%%%%

% Najde maximalni parovani grafu Graph, vysledek vraci v Result
maximum_matching(Graph,Result) :- augmenting_path(Graph), copy_graph(Graph,Copy), maximum_matching(Copy,Result).
maximum_matching(Graph,Result) :- contract(Graph,ContrGraph), maximum_matching(ContrGraph,NewContrGraph), decontract(NewContrGraph,Result).
maximum_matching(Graph,Graph).

% Vic user-friendly: zaroven nacte graf ze souboru Filename
find_maximum_matching(Filename) :- load_graph(Filename,Graph), maximum_matching(Graph,Result), print_graph(Result), print_pairs(Result).

%%%%%%%%%% DEBUG %%%%%%%%%%
print_graph([H|T]) :- notcontr(H), nodenum(H,V), write(V), write(': '), edges(H,E), print_neighb(E), print_graph(T),!.
print_graph([H|T]) :- print_node(H), write('-> '), realcontr(H,C), print_node(C), write('\n'), print_graph(T),!.
print_graph([]) :- write('\n').

print_neighb([H|T]) :- print_node(H), print_neighb(T).
print_neighb([]) :- write('\n').

print_node(N) :- nodenum(N,V), write(V), write(' ').

print_pairs([H|T]) :- notcontr(H), paired(H), pair(H,P), nodenum(H,V1), nodenum(P,V2), V1 < V2, write(V1), write('-'), write(V2), write('\n'), print_pairs(T), !.
print_pairs([_|T]) :- print_pairs(T), !.
print_pairs([]) :- write('\n').
