/* Hledani maximalniho parovani v grafu pomoci Edmondsova algoritmu *\
 * Graf se zadava na standardnim vstupu v tomto tvaru:              *
 *   Na prvni radce je pocet vrcholu a pocet hran oddelene mezerou  *
 *   Na kazdem nasledujicim radku je hrana (cisla vrcholu oddelena  *
 *     mezerou, cisla jsou od 0 !!!)                                *
 * Vystup: hrany patrici do nalezeneho parovani                     *
\*                     Lukas Turek 2005                             */

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <assert.h>

#define NONE -1

//Spojovy seznam hran
typedef struct tagEDGE { int end; struct tagEDGE *next; } EDGE;

//Vrchol: seznam hran, s cim je sparovany, jestli je zkontrahovany
typedef struct { EDGE *edges; int pair; int contr; } NODE; 

//Seznam vrcholu (pro ulozeni nalezene kruznice)
typedef struct tagNODELIST { int node; struct tagNODELIST *next; } NODELIST;

int n,m; //Pocet vrcholu a hran
NODE *graph; //Zadany graf


//Pridani (orientovane) hrany mezi vrcholy X a Y do grafu 
void add_edge(int x, int y)
{
  EDGE *new;
  
  assert( new = (EDGE*)malloc(sizeof(EDGE)) );
  new->end = y;
  new->next = graph[x].edges;
  graph[x].edges = new;
}


//Vymaze hranu z grafu 
//(spojovy seznam je jednosmerny, takze musim pouzit dvojity pointer)
void del_edge(EDGE **pp)
{
  EDGE *p = *pp;
  *pp = p->next;
  free(p);
}


//Sparovani vrcholu U a V
void pair(u,v)
{
  graph[u].pair = v;
  graph[v].pair = u;
}


//Pridani vrcholu NODE do seznamu vrcholu LIST
void add_to_list(int node, NODELIST **list)
{
  NODELIST *p;
  
  assert( p = (NODELIST*)malloc(sizeof(NODELIST)) );
  p->node = node;
  p->next = *list;
  *list = p;
}


//Vrati posledni prvek seznamu vrcholu
NODELIST *last(NODELIST *list)
{
  NODELIST *q = list;
  
  assert(q);
  
  while( q->next )
    q = q->next;
  
  return q;
}


//Dealokace _cyklickeho_ seznamu vrcholu
void free_list(NODELIST *list)
{
  NODELIST *p = list, *tmp;
  
  do {
    tmp = p->next;
    free(p);
    p = tmp;
  } while(p != list);
}


//Nalezeni, ktery vrchol kruznice ma za souseda zadany vrchol
NODELIST *find_paired_node(NODELIST *circle, int node)
{
  EDGE *p;
  NODELIST *q;
  
  /* Projde se kruznice, u kazdeho vrcholu vsichni jeho sousede, */
  /* hledame ktery vrchol kruznice ma za souseda zadany vrchol   */
  for(q = circle; q; q = q->next)
    for(p = graph[q->node].edges; p; p = p->next)
      if(p->end == node)
        return q;  
  
  return NULL;
}


//Hledani volne stridave cesty (VSC) z uzlu NODE 
//(rekurzivne, predava se vrstva LAYER a pole oznaceni MARK)
//vraci jestli se povedlo najit VSC
int dfs_path(int node, int layer, char *mark)
{
  EDGE *p; //Hrana vedouci z vrcholu
  mark[node] = 1; //Oznacime navstiveny vrchol
  
  if(layer) //V liche vrstve jdeme po parove hrane
    return dfs_path(graph[node].pair, !layer, mark);
  
  //Prochazeni hran z vrcholu
  for(p = graph[node].edges; p; p = p->next)
  {
    //Je-li soused nesparovany, nebo pres nej vede VSC
    if( !mark[p->end] && 
        (graph[p->end].pair == NONE || 
         dfs_path(p->end, !layer, mark) )
      )
    {
      pair(p->end, node); //Prehodime parovani na ceste
      return 1; //Oznameni, ze jsme nasli VSC
    }
  }
  
  return 0;
}


//Hledani liche kruznice ("kvetu")
//(rekurzivne, predava se vrstva LAYER a pole oznaceni MARK, vraci kruznici v CIRCLE)
//Vraci jestli vrchol lezi na kruznici
int dfs_circle(int node, int layer, char *mark, NODELIST **circle)
{
  EDGE *p; //Hrana vedouci z vrcholu
  mark[node] = 1 + layer; //Vrcholy se znaci podle vrstvy
  
  if(layer) //V liche vrstve jdeme po parove hrane
    //Pokud parovy vrchol lezi na liche kruznici
    if( dfs_circle(graph[node].pair, !layer, mark, circle) )
    {
      //Vrchol na kruznici se prida do seznamu a oznaci
      add_to_list(graph[node].pair, circle);
      mark[graph[node].pair] = 3;
      return 1;
    }
    else
      return 0;
    
  //Prochazeni hran z vrcholu
  for(p = graph[node].edges; p; p = p->next)
  {
    /* Kruznici jsme nasli, pokud hrana vede do navstiveneho vrcholu */
    /* na sude vrstve a nebo sousedni vrchol lezi na kruznici        */
    if( mark[p->end] == 1 || 
        ( !mark[p->end] && dfs_circle(p->end, !layer, mark, circle) ) )
    {  
      //Vrchol na kruznici se prida do seznamu a oznaci
      add_to_list(p->end, circle);
      mark[p->end] = 3;
      
      //Pokud byl tento vrchol oznacen ze lezi na kruznici, tak kruznice konci
      return mark[node] != 3; 
    }
    
    if(*circle) return 0;
  }
   
  return 0;
}


//Kontrakce kruznice CIRCLE, potrebuje oznaceni vrcholu MARK
void contract(NODELIST *circle, char *mark)
{
  EDGE *p,*q,**pp;
  
  n++; //Pribude vrchol, do nehoz se kruznice zkontrahuje
  
  while(circle) //Prochazeni kruznice
  { //Prochazeni sousedu vrcholu na kruznici
    for(p = graph[circle->node].edges; p; p = p->next)
    { 
      if(mark[p->end] < 3) //Soused nelezi na kruznici 
      {
        mark[p->end] = 4; //Kazdeho souseda pridavame jen jednou
    
        /* Prochazeni hran ze souseda, ruseni tech do kruznice */
        /* Varovani: dvojite pointery, poradne se posadte :-)  */
        pp = &graph[p->end].edges;
        while( (q = *pp) )
        {
          if(mark[q->end] == 3)
            del_edge(pp);
          else
            pp = &q->next;
        }
        
        add_edge(p->end, n - 1); /* Pridame hranu z noveho */
        add_edge(n - 1, p->end); /* vrcholu do souseda     */
      }
    }
    
    pair(graph[circle->node].pair, n - 1); //Kontrahují se i páry vrcholů
    graph[circle->node].contr = 1; //Oznaceni, ze je vrchol kontrahovan
    
    circle = circle->next;
  }
}


//Dekontrakce vrcholu na kruznici CIRCLE
void decontract(NODELIST *circle)
{
  int pair_edge = 0; //Jestli ma byt hrana sparovana
  NODELIST *q, *prev; //Vrcholy na kruznici
  EDGE *p, *tmp; //Seznamy hran
  
  /* Kruznice zacina vrcholem, kterym do ni vede stridava cesta ("stonek"), */
  /* Tento vrchol bude sparovan s tim, s cim byl sparovan pridany vrchol    */
  pair(circle->node, graph[n-1].pair);
  
  q = circle->next;
  prev = circle;
  
  do { //Prochazeni kruznice
    //Prochazeni hran z vrcholu na kruznici
    for(p = graph[q->node].edges; p; p = p->next)
    { 
      //Pokud soused nelezi na kruznici
      if( p->end != prev->node && p->end != (q->next)->node )
      {  
        /* Pridana hrana se odstrani, pokud jsme to  */
        /* jeste neudelali (v seznamu je vzdy prvni) */
        if( graph[p->end].edges->end == n-1 )
          del_edge(&graph[p->end].edges);
        
        //Pridame predtim odstranenou hranu
        add_edge(p->end, q->node);
      }      
    }
      
    //Hrany se stridave paruji
    if(pair_edge)
      pair(prev->node, q->node);
    pair_edge = !pair_edge;
    
    //Nezapomenout zrusit oznaceni, ze je vrchol kontrahovan
    graph[q->node].contr = 0;
    
    prev = q;
    q = q->next;
  } while(q != circle->next);
  
  /* Zruseni pridaneho vrcholu     */
  /* (uvolneni seznamu hran z nej) */
  p = graph[n-1].edges;
  graph[n-1].edges = NULL;
  graph[n-1].pair = NONE;
  
  while(p)
  {
    tmp = p->next;
    free(p);
    p = tmp;
  }
  
  n--;
}


//Nacteni grafu (ze standardniho vstupu)
void load_graph(void)
{
  int i;
  int x,y;
  
  scanf("%d %d", &n, &m); //Pocet vrcholu a hran
  assert( graph = (NODE*)malloc(2 * n * sizeof(NODE)) );
  
  //Vynulovani pole
  for(i = 0; i < n; i++)
  {
    graph[i].edges = NULL;
    graph[i].pair = NONE;
    graph[i].contr = 0;
  }
  
  //Nacteni hran
  for(i = 0; i < m; i++)
  {  
    scanf("%d %d", &x, &y);
    add_edge(x,y);
    add_edge(y,x);
  }
}


//Vypis grafu
void print_graph(void)
{
  int i;
  EDGE *p;
  
  for(i = 0; i < n; i++)
    if(!graph[i].contr)
    {  
      printf("%d: ", i);
      p = graph[i].edges;
      
      while(p)
      {
        printf("%d ", p->end);
        p = p->next;
      }
      
      putchar('\n');
    }
}


//Vypis paru
void print_pairs(void)
{
  int i;
  
  for(i = 0; i < n; i++)
  {
    if(graph[i].pair != NONE && i < graph[i].pair)
      printf("%d-%d\n", i, graph[i].pair);
  }
}


//Hledani maximalniho parovani
void max_pair()
{
  int i;
  NODELIST *circle = NULL, *q;
  char *mark; //Oznaceni navstivenych vrcholu
  
#ifdef DEBUG
  print_graph();
  print_pairs();
  puts("==========================");
#endif
  
  assert( mark = (char*)malloc(n) );
  
  //Napred vyhledame vsechny VSC
  memset(mark, 0, n);
  for(i = 0; i < n; i++)
    if( graph[i].pair == NONE && !mark[i] && !graph[i].contr )
      dfs_path(i, 0, mark);
  
  //Pak zkusime najit lichou kruznici
  memset(mark, 0, n);
  for(i = 0; i < n; i++)
    if( graph[i].pair == NONE && !mark[i] && !graph[i].contr )
    {
      dfs_circle(i, 0, mark, &circle);
      if(circle) break;
    }
  
  if(circle)
  {
    contract(circle, mark); //Zkontrahujeme kruznici
    free(mark); //Setrime pameti
    
    max_pair(); //Rekurze na zkontrahovany graf
    
    /* Hledame "stonek": ktery vrchol kruznice je sparovan s parem pridaneho */
    /* vrcholu (ostatni vrcholy kruznice jsou sparovane vramci kruznice)     */
    q = find_paired_node(circle, graph[n-1].pair);
    
    //Udelame kruznici opravdu kruznici (cyklicky seznam)
    last(circle)->next = circle;
    
    decontract(q); //Dekontrakce zacne od "stonku"
    free_list(circle); //Uvolneni kruznice (dfs_circle ji alokuje dynamicky)
  }  
}


//---<MAIN>---
int main()
{
  load_graph(); //Nacteni grafu
  max_pair(); //Nalezeni parovani
  print_pairs();  //Vypis parovani
  return 0;
}
