/* Tetris - game engine *\
\*  Lukas Turek, 2005   */

//Standardni (ANSI) knihovny
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "tetris.h"

//Parametry definujici zpusob vyberu kostek
int bricksel = BRSEL_RANDOM; //Zpusob, jak se vybira kostka
double log_a; //Logaritmus intenzity (intenzitu pro vypocet nepotrebujeme)
double scale; //Koeficient pro prevod z nahodneho cisla do intervalu pravdepodobnosti
unsigned long seed = 1;

//Generovani nahodnych cisel (linearni kongruencni generator)
//Generuje cisla z intervalu 0..(max-1), max muze byt nejvyse 32768
unsigned int myrand(int max) 
{
  seed = seed * 1103515245 + 12345;
  return (seed >> 16) % max;
}

//Nastaveni pocatecni hodnoty nahodneho generatoru
void mysrand(unsigned long newseed) 
{
  seed = newseed;
}

//Vrati v C zvolenou pozici kostky BRICK, navratova hodnota je jeji vyhodnost
int find_pos(T_BRICK *brick, T_COORD *c, T_GAME *game)
{
  int i,j; //Index cyklu, pomocna promenna
  int r,x,y; //Testovane souradnice kostky
  int start, end; //Souradnice x zacatku a konce kostky
  int sum; //Soucet vysek - pro urceni poctu der
  int new_h[4]; //Vyska sloupce po dopadu kostky na testovanou pozici
  long qual; //Vyhodnost pozice
  long best_qual = LONG_MIN; //Vyhodnost zatim nejlepsi pozice
  T_SHAPE *shape; //Kostka ve zvolenem natoceni

  for(r = 0; r < brick->n_rot; r++) //Hledani nejvyhodnejsi pozice kostky
  {
    shape = &brick->rot[r]; //Ulozeni otocene kostky pro zjednoduseni
    start = shape->x1;   /* Zjisteni, kde kostka zacina */
    end = shape->x2 + 1; /* a konci ve vodorovnem smeru */

    for(x = 1 - start; x < 12 - end; x++) /* Test vsech pozic kostky */
    {                                     /* v tomto otoceni         */
      sum = 0;
      y = 25;
      for(i = start; i < end; i++) //Hledani vysky dopadu kostky
      { //Kostka se zarazi ve vysce sloupce + vyska kostky
        j = game->height[x + i] + shape->col_height[i];
        sum += j; //Scitani vysek pro hledani der
        if(24 - j < y) /* Kostka se zarazi  */
          y = 24 - j;  /* na nejvyssim bode */
      }

      if(y >= 0) //Muze-li sem kostka dopadnout
      { //Za diru se odecita COEF1 bodu, za vysku bod/ctverecek
        qual = COEF1 * (sum - (end - start) * (24 - y)) + y;

        for(i = 0; i < 4; i++) //Hledaji se zaplnene rady
          if(shape->row_width[i] + game->row_sq[y + i] == 10)
            qual += COEF2;  //Za zaplnenou radu se pricita COEF2 bodu

        for(i = 3; i >= 0; i--) //Zjistuji se vysky sloupcu po dopadu kostky
          new_h[shape->sq[i].x] = 24 - y - shape->sq[i].y;

        //Vypocita se rozdil mezi vyskami vsech sloupcu a zjisti jeho zmena po
        //padu kostky. Za kazdou jednotku vysky nad 2 se odecita COEF3 bodu.
        if((start + x) > 1) //Neni kostka u kraje ?
        { //Rozdil vysky sloupce pred kostkou a 1. sloupce kostky
          j = abs(new_h[start] - game->height[start + x - 1]); /* Pomocna  */
          if(j > 2) //Rozdil pod tri nevadi                    /* promenna */
            qual -= COEF3 * (j - 2);
        }
        if((end + x) < 11) //Neni kostka u kraje ?
        { //Rozdil vysky posledniho sloupce kostky a nasledujiciho sloupce
          j = abs(new_h[end - 1] - game->height[x + end]);
          if(j > 2)
            qual -= COEF3 * (j - 2);
        }

        //Nyni se projdou pozice pod kostkou, pocota se byvaly rozdil vysek
        for(i = x + start; i < x + end + 1; i++)
          if(i > 1 && i < 11) //Nejsme-li u kraje
          { //Rozdil puvodnich vysek
            j = abs(game->height[i] - game->height[i-1]);
            if(j > 2)                  /* Za byvaly rozdil se */
              qual += COEF3 * (j - 2); /* body naopak prictou */
          }

        if(qual > best_qual) /* Kdyz je toto zatim nejvyhodnejsi pozice  */
        {                    /* nebo je zacatek cyklu anebo jeste nebyla */
          best_qual = qual;  /* nalezena zadna jina pozice               */
          if(c) //Neni-li C NULL, zapiseme do nej nalezenou pozici
          {
            c->x = x;
            c->y = y;
            c->r = r;
          }
        }

      } //Konec if(y>=0)

    } //Konec prochazeni pozic
  } //Konec prochazeni otoceni

  return best_qual; //Vraci se vyhodnost vybrane pozice
}


//Vymaze zaplnene rady (hleda je mezi Y a END) a vrati jejich pocet
int full_rows(int y, int end, T_GAME *game)
{
  int x; //Index cyklu (ctverecek v rade)
  int _y = y; //Rada, na kterou se ma rada Y posunout
  int del_rows = 0; //Kolik rad se vymazalo

  game->last_changed = -1; //Zadne rady se jeste nezmenily

  /* Rady, kam se polozila kostka se projdou vsechny, zbytek jen pokud  */
  /* se ma posouvat, je co posouvat (nebo mazat) a nejsme na konci pole */
  while(y >= end || (del_rows && _y >= 0 && game->row_sq[_y]))
  {
    if(game->row_sq[y] == 10) //Rada je zaplnena - 10 ctverecku
    {
      del_rows++; //Dalsi rada se bude posouvat dolu
      if(game->last_changed == -1)
        game->last_changed = y;
    }
    else if(del_rows > 0) //Neni-li rada plna a je co posouvat
    {
      if(y >= 0) /* Rada Y se presune na _Y, */
      {          /* pokud rada Y existuje    */
        for(x = 1; x < 11; x++)
          game->board[x][_y] = game->board[x][y];
        game->row_sq[_y] = game->row_sq[y]; //Posouva se i pocet ctverecku
      }
      else /* Jinak se na rada _Y vymaze             */
      {    /* (jakoby se na ni posouva prazdna rada) */
        for(x = 1; x < 11; x++) //Rada se maze
          game->board[x][_y] = 0;
        game->row_sq[_y] = 0;
      }
    }

    y--; //Jdeme na dalsi radu (na obrazovce nahoru)
    _y = y + del_rows; /* Rada, na kterou se ma rada Y presouvat */
  }                    /* se uklada pro zjednoduseni             */

  for(x = 1; x < 11; x++) /* Po vymazani rad(y) se musi */
  {                       /* vysky prislusne snizit     */
    game->height[x] -= del_rows; //Snizeni o vymazane rady
    while(!game->board[x][24 - game->height[x]]) /* Ale pod vymazanou    */
      game->height[x]--;                         /* radou mohla byt dira */
  }

  //Nastavuje se prvni (ale ze smeru prochazeni posledni) zmenena rada
  game->first_changed = y < 0 ? 0 : y + 1;

  return del_rows; //Funkce vraci pocet vymazanych rad (pro statistiku)
}


//Posune kostku BRICK na souiradnicich C dle prikazu CMD, pokud se nebude
//prekryvat se zaplnenymi ctverecky v poli BOARD.
//Vraci 0: kostka dopadla, 1: kostka se neposunula, 2: kostka se posunula
int move_brick(int cmd, T_COORD *c, T_BRICK *brick, T_BOARD board)
{
  int i; //Index cyklu
  T_COORD _c = *c; //Puvodni pozice kostky

  switch(cmd)
  {
    case CMD_LEFT: //Doleva
      c->x--;
      break;
    case CMD_RIGHT: //Doprava
      c->x++;
      break;
    case CMD_DOWN: //Dolu
      c->y++;
      break;
    case CMD_RROT: //Otoceni po smeru hodinovych rucicek
      c->r = (c->r < brick->n_rot - 1) ? c->r + 1 : 0;
      break;
    case CMD_LROT: //Otoceni proti smeru hodinovych rucicek
      c->r = c->r ? c->r - 1 : brick->n_rot - 1;
      break;
    case CMD_FALL: //Pad kostky: posun dolu, dokud to jde
      while(move_brick(CMD_DOWN, c, brick, board));
      return 0;
    default: 
      return 1;
  }

  //Testovaci cyklus - jestli se kostka neprolina s ostatnimi
  for(i = 0; i < 4; i++)
    if(board[c->x + brick->rot[c->r].sq[i].x]
            [c->y + brick->rot[c->r].sq[i].y])
    {
      *c = _c; //Pri pozitivnim testu navrat k puvodni pozici
      if(cmd == CMD_DOWN) //Vyjimka - pri pohybu dolu
        return 0; //Kostka byla polozena
      else
        return 1; //Pozice kostky zustala stejna, ale jinak je vse normalni
    }

  return 2; //Pozice kostky byl a uspesne zmenena
}

//Polozi kostku SHAPE na souradnice C, aktualizuje statistiku STATS,
//vraci 1, pokud se kostku povedlo polozit, jinak 0 (zaplneno, konec)
int set_brick(T_SHAPE *shape, T_COORD *c, T_GAME *game)
{
  int i; //Index cyklu
  int del_rows; //Kolik rad se vymazalo

  if(c->x == 4 && c->y == 0) //Kostka zustala ve vychozi pozici -> zaplneno
    return 0; //Ukonci program

  for(i = 0; i < 4; i++) /* Zapis kostky do pole BOARD           */
  {                      /* a aktualizace poctu ctverecku v rade */
    game->board[c->x + shape->sq[i].x][c->y + shape->sq[i].y] = 1;
    game->row_sq[c->y + shape->sq[i].y]++;
  }

  for(i = 0; i < 4; i++) //Aktualizace vysek sloupcu
    if(game->height[c->x + shape->sq[i].x] < 24 - (c->y + shape->sq[i].y))
      game->height[c->x + shape->sq[i].x] = 24 - (c->y + shape->sq[i].y);

  //Mazani zaplnenych rad
  del_rows = full_rows(c->y + shape->sq[3].y, c->y + shape->sq[0].y, game);

  game->stats.all_rows += del_rows; //Pocet vymazanych rad - statistika
  game->stats.rows_by_number[del_rows]++; //Pocet vymazani po ... radach

  return 1;
}


//Nacte kostky ze vstupniho souboru do pole BRICKS, vraci, zda se to povedlo
int read_bricks(T_BRICK *bricks)
{
  int b, r; //Cislo kostky a jeji natoceni
  int i; //Index cyklu
  int x, y; //Nactene souradnice ctverecku
  FILE *fr; //Soubor s daty
  T_SHAPE *shape; //Kostka v aktualnim otoceni

  if(!(fr = fopen("tetris.dat", "rb"))) /* Otevirani s kontrolou     */
    return 0;                           /* (0 znamena chybu - FALSE) */

  for(b = 0; b < 7; b++) //Nacitani jednotlivych kostek
  {
    bricks[b].num = b; //Cislo kostky (z brick* by jinak neslo zjistit)
    
    fscanf(fr, "%d", &bricks[b].n_rot);  /* Nacte se pocet otoceni    */
    for(r = 0; r < bricks[b].n_rot; r++) /* a pak se otoceni prochazi */
    {
      shape = &bricks[b].rot[r]; //Natocena kostka  se uklada pro zjednoduseni

      memset(shape->row_width, 0, 4);
      memset(shape->col_height, 0, 4);

      for(i = 0; i < 4; i++) //Nacitani ctverecku kostky
      {
        fscanf(fr, "%d", &x);
        fscanf(fr, "%d", &y);
        shape->sq[i].x = x; /* Souradnice ctverecku */
        shape->sq[i].y = y; /* se zapisuji          */

        shape->row_width[y]++;  //Pocitaji se ctverecky v jednotlivych radach

        //Vypocet vysek jednotlivych sloupcu kostky
        if(shape->col_height[x] < y + 1) /* Hleda se v kazdem sloupci */
          shape->col_height[x] = y + 1;  /* x ctverecek s nejvetsim y */
      }

      shape->x1 = (shape->col_height[0] == 0);
      shape->x2 = shape->x1 - 1;
      for(i = 0; i < 4; i++)     /* Pocita se sirka kostky pomoci souctu  */
        if(shape->col_height[i]) /* sloupcu, v nichz je nejaky ctverecek, */
          shape->x2++;           /* tj vyska soupce je nenulova           */

      shape->y1 = shape->sq[0].y; /* Ctverecky kostky jsou    */
      shape->y2 = shape->sq[3].y; /* serazany vzestupne dle Y */
    }
  }

  fclose(fr);
  return 1;
}


//Inicializace struktury GAME
void initialize(T_GAME *game)
{
  int i; //Index_cyklu;

  memset(game, 0, sizeof(T_GAME));

  for(i = 0; i < 12; i++)   //
    game->board[i][24] = 1; //Ramecek
                            //
  for(i = 0; i < 25; i++)   //okolo
  {                         //
    game->board[0][i] = 1;  //
    game->board[11][i] = 1; //pole
  }                         //

  game->first_changed = -1; /* Oznaceni, ze se zadne */
  game->last_changed = -1;  /* rady nezmenily        */
}


//Vrati pozici K-teho nejvetsi prvek v poli DATA velikosti N
int k_th_largest(long *data, int n, int k)
{
  int i;
  int max = LONG_MIN, pos = 0;

  while(k--)
  {
     max = LONG_MIN;
     for(i = 0; i < n; i++)
       if(data[i] > max)
       {
         max = data[i];
         pos = i;
       }
     data[pos] = LONG_MIN;
  }
  return pos;
}


//Vrati nove vybranou kostku a aktualizuje statistiku ve strukture GAME
T_BRICK *new_brick(T_BRICK *bricks, T_GAME *game)
{
  int n; //Cislo kostky

  if(bricksel == BRSEL_RANDOM) //Nahodny vyber kostek
    n = myrand(7);
  else
  {
    int k; //Kolikata nejvyhodnejsi kostka se vybere
    long qual[7]; //Vyhodnosti jednotlivych kostek
    
    /* Pocitaji se vyhodnosti jednotlivych kostek (pokud vybirame horsi     */
    /* kostku, zapisujeme opacne hodnoty, abychom zase hledali vetsi cislo) */
    if(bricksel == BRSEL_BETTER)
      for(n = 0; n < 7; n++)
        qual[n] = find_pos(bricks + n, NULL, game);
    else
      for(n = 0; n < 7; n++)
        qual[n] = -find_pos(bricks + n, NULL, game);

    /* Vygeneruje se nahodne cislo, spocita se, do kolikateho useku patri,   */
    /* (7 - cislo useku) pak urcuje, kolikata nejvyhodnejsi kostka se vybere */
    k = 7 - (int)(log(myrand(7) * scale + 1.0) / log_a);
    n = k_th_largest(qual, 7, k); //Vyber k-tou kostku podle kvality
  }
  
  game->stats.all_bricks++; //Celkovy pocet kostek
  game->stats.single_brick[n]++; //Pocet dopadnutych kostek cislo N
  return bricks + n;
}


//Nastaveni zpusob vyberu kostek (druhu a intenzity)
void set_bricksel_mode(int brsel_mode, int brsel_intens)
{
  double a; /* Koeficient urcujici rozlozeni pravdepodobnosti   */
            /* (1. kostka ma pravdepodobnost A, druha A^2, ...) */

  bricksel = brsel_mode; //Uklada se zpusob vyberu kostky
  if(bricksel != BRSEL_RANDOM) //Pokud ma byt vyber ovlivneny situaci
  {
    //Zadana intensita se z (1..20) prevede do (1.1 .. 3.0)
    a = (double)brsel_intens / 10.0 + 1.0;
    //Pocita se koeficient pro prevod z nahodneho cisla do intervalu pravdepodobnosti
    scale = (pow(a,7) - 1);
    //Predpocita se logaritmus A (samotne A dale nepotrebujeme)
    log_a = log(a);
  }
}

