#include <string.h>
#include <errno.h>

#include <unistd.h>
#include <sys/types.h> 
#include <sys/poll.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "tetris.h"
#include "tetris-net.h"

//Ty paketu (prvni byte)
#define DGRAM_QUIT        0
#define DGRAM_START       1
#define DGRAM_SEED        2
#define DGRAM_BRICKPOS    3
#define DGRAM_BOARD       4
#define DGRAM_STATS       5

#define BUFFSIZE        512
#define DGRAM_BOARD_SIZE 39

int sockfd; //UDP socket pro komunikaci s druhou stranou
struct sockaddr remote_addr; //Adresa druhe strany


//Pripojeni k serveru (port muze byt NULL, pak se pouzije default)
const char *net_init_client(char *host, char *port)
{
  int error;
  struct addrinfo hints, *ai;
  
  memset(&hints, 0, sizeof(hints));
  hints.ai_family = AF_INET; //TCP/IP socket
  hints.ai_socktype = SOCK_DGRAM; //UDP

  //Resolv DNS jmena - Nahrada za gethostbyname()
  error = getaddrinfo(host, port ? port : DEFAULT_PORT, &hints, &ai);
  if(error) return gai_strerror(error);
  
  sockfd = socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol);
  if(sockfd < 0) 
    return strerror(errno);

  remote_addr = *(ai->ai_addr); //Adresa se uklada kvuli sendto()
  freeaddrinfo(ai);
  return 0;
}


//Spusteni serveru (vraci se hned, port muze byt NULL, pak se pouzije default)
const char *net_init_server(char *port)
{
  int error;
  struct addrinfo hints, *ai;
  
  memset(&hints, 0, sizeof(hints));
  hints.ai_family = AF_INET; //TCP/IP
  hints.ai_socktype = SOCK_DGRAM; //UDP

  //Posloucha na vsech adresach
  error = getaddrinfo("0.0.0.0", port ? port : DEFAULT_PORT, &hints, &ai);
  if(error) return gai_strerror(error);
  
  sockfd = socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol);
  if(sockfd < 0) 
    return strerror(errno);
  
  if(bind(sockfd, ai->ai_addr, ai->ai_addrlen) < 0) 
    return strerror(errno);
  
  freeaddrinfo(ai);
  return 0;
}


//Poslani cisla a pozice kostky
void net_send_brick(int bricknum, T_COORD *c)
{
  char buffer[5];
  
  if(sockfd == -1) return;
  
  buffer[0] = DGRAM_BRICKPOS;
  buffer[1] = bricknum;
  buffer[2] = c->r;
  buffer[3] = c->x;
  buffer[4] = c->y;
  
  sendto(sockfd, buffer, sizeof(buffer), 0, &remote_addr, sizeof(remote_addr));
}
      

//Poslani stavu hraciho pole
void net_send_board(T_GAME *game)
{
  int i;
  char buffer[DGRAM_BOARD_SIZE]; // 12 * 25 / 8
  
  memset(buffer, 0, DGRAM_BOARD_SIZE);
  buffer[0] = DGRAM_BOARD;
  
  //Stavy ctverecku se posilaji v bitech kvuli uspore mista
  for(i = 0; i < 25 * 12; i++)
    buffer[(i >> 3) + 1] |= game->board[i%12][i/12] << (i & 7);

  sendto(sockfd, buffer, sizeof(buffer), 0, &remote_addr, sizeof(remote_addr));
}


//Poslani poctu vymazanych rad
//Vyzaduje sizeof(short) == 2 !!!
void net_send_stats(T_STATS *stats)
{
  int i;
  short n;
  char buffer[9]; // 1 + 4 * sizeof(short)
        
  buffer[0] = DGRAM_STATS;
  
  //Pozor: rows_by_number je 1..4 !
  for(i = 0; i < 4; i++)
  {  
    n = htons(stats->rows_by_number[i+1]);
    memcpy(buffer + 1 + 2 * i, (char*)&n, 2);
  }

  sendto(sockfd, buffer, sizeof(buffer), 0, &remote_addr, sizeof(remote_addr));
}


//Ukonceni hry (jen 1 byte)
void net_send_quit()
{
  char buffer[1];
  
  buffer[0] = DGRAM_QUIT;
  sendto(sockfd, buffer, sizeof(buffer), 0, &remote_addr, sizeof(remote_addr));
}


//Prijeti + zpracovani dat
int net_receive(T_GAME *game, int *bricknum, T_COORD *c)
{      
  int i;
  int size;
  char buffer[BUFFSIZE];
  int flags = 0; //Bitova maska co se zmenilo
  struct pollfd pfd = { sockfd, POLLIN, 0 };
  
  while(poll(&pfd,1,0) > 0) //Dokud prichazi data
  {  
    size = recv(sockfd, buffer, BUFFSIZE, 0);
  
    switch(buffer[0])
    {
      case DGRAM_BRICKPOS: /* Souradnice kostky   */
        if(size == 5)      /* NEPROVADI KONTROLU! */
        {
          *bricknum = buffer[1];
          c->r = buffer[2];
          c->x = buffer[3];
          c->y = buffer[4];
          flags |= FLAG_BRICK;
        }
        break;
        
      case DGRAM_BOARD: /* Stav hraciho pole */
        if(size == DGRAM_BOARD_SIZE)
        {
          for(i = 0; i < 25 * 12; i++) /* V kazdem jazyce se da programovat v Assembleru... */
            game->board[i%12][i/12] = !!(buffer[(i >> 3) + 1] & (1 << (i & 7)));
          flags |= FLAG_BOARD;
        }
        break;
      
      case DGRAM_STATS: /* Pocty smazanych rad */
        if(size == 9)
        {          
          short n;
          
          game->stats.all_rows = 0;
          for(i = 0; i < 4; i++) /* Berou se pocty smazanych n-tic, */
          {                      /* pocita se celkovy pocet rad     */
            memcpy((char*)&n, buffer + 1 + 2 * i, 2);
            game->stats.rows_by_number[i+1] = ntohs(n);
            game->stats.all_rows += ntohs(n) * (i+1);
          }
          
          flags |= FLAG_STATS;
        }
        break;
      
      case DGRAM_QUIT: /* Konec hry */
        flags |= FLAG_END;
        break;
    }
  }
  
  return flags;
}


//Server ceka na klienta, ulozi jeho IP, vraci random seed ktere poslal
unsigned int net_wait_server()
{
  int size;
  char buffer[BUFFSIZE];
  unsigned long seed;
  int addrlen = sizeof(remote_addr);
  
  do { //Ceka se na pocatecni paket se seed pro nahodny generator
    size = recvfrom(sockfd, buffer, BUFFSIZE, 0, &remote_addr, &addrlen);
  } while(size != 5 || buffer[0] != DGRAM_SEED);
    
  memcpy((char*)&seed, buffer+1, 4); //Vytahne se random seed z paketu
  
  buffer[0] = DGRAM_START; //Odpoved: hra muze zacit
  sendto(sockfd, buffer, 1, 0, &remote_addr, sizeof(remote_addr));
  
  return ntohl(seed); //Random seed se posila jako big-endian
}


//Klient se pripoji na server a ceka, posila random seed
void net_wait_client(unsigned long seed)
{
  int size;
  char buffer[BUFFSIZE];
  
  buffer[0] = DGRAM_SEED;
  seed = htonl(seed); //Random seed se posila jako big-endian
  memcpy(buffer+1, (char*)&seed, 4); 
  sendto(sockfd, buffer, 5, 0, &remote_addr, sizeof(remote_addr));
  
  do { //Cekani na odpoved od serveru
    size = recv(sockfd, buffer, BUFFSIZE, 0);
  } while(size != 1 || buffer[0] != DGRAM_START);
}
